-- [SYNCRO-TABLE] Schlüsselbegriffe für jeden Datensatz
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Picndoku
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Dokumentenmanagemenet_-_DMS
CREATE TABLE recnocommentkategorie (
  rck_id                varchar(30) PRIMARY KEY,  -- [SYNCRO:SyncID]
  rck_schema            varchar(50),              -- Schema wo die Tabele aus rck_id liegt (für Verlinkung über ab_tablename/ab_keyvalue)
  rck_modified          timestamp(0),             -- [SYNCRO:Modified]
  rck_b                 varchar(30) UNIQUE,
  rck_gruppe            varchar(20),
  rck_column            varchar(20),
  rck_mapto             varchar(30),              -- RechNr ist ein Schlüsselwort für Einkauf und Verkauf. Daher muß belkopf.be_bnr auf rechnr gemappt werden

  -- CHECK for existing schema
  CONSTRAINT recnocommentkategorie__schema__existing CHECK (
         rck_schema IS NULL
      OR tsystem.schema_exists( rck_schema )
  )

);

CREATE OR REPLACE FUNCTION recnocommentkategorie__b_iu() RETURNS TRIGGER AS $$
  BEGIN
    IF tg_op='UPDATE' AND current_user<>'syncro' THEN
        new.rck_modified := currenttime();
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER recnocommentkategorie__b_iu
    BEFORE INSERT OR UPDATE
    ON recnocommentkategorie
    FOR EACH ROW
    EXECUTE PROCEDURE recnocommentkategorie__b_iu();
--

-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Picndoku
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Dokumentenmanagemenet_-_DMS
-- Achtung! Bei Tabellenstrukturänderung, die Funktionen von #7135 sollen angepasst werden
CREATE TABLE recnokeyword (
  r_id                  serial PRIMARY KEY,                                             -- [SYNCRO:NotThisFields]
  r_tablename           varchar(50),                                                    -- zugehörige Haupttabelle, z.B. adk/art
  r_dbrid               varchar,                                                        -- entweder in picndoku bei Dokumente oder bei Eigenschaften zB in art
  r_kategorie           varchar(30) REFERENCES recnocommentkategorie ON UPDATE CASCADE, -- z.B. Oberfläche
  r_descr               varchar(100),                                                   -- Wert, z.B. Farbe
  r_unit                varchar(50),                                                    -- Maßeinheit frei zb
  r_txt                 text,                                                           -- Zusatztext für Beschreibung
  -- nächste Felder sind für Artikelparameter
  r_not                 bool NOT NULL DEFAULT FALSE,
  r_value               varchar(100),                                                   -- z.B. blau
  r_min                 numeric(12,4),
  r_max                 numeric(12,4),
  r_reg_pname           varchar(40),                                                    -- nachgebauter CONSTRAINT auf recnogroup.reg_pname, siehe recnogroup__a_ud__reg_pname__forbid_if_used
  -- System (tables__generate_missing_fields)
  --   kein automatischer table_delete-Trigger (tables__fieldInfo__fetch) ... wäre hier auch sinnlose, da keine Keywords an der KeywordTabelle hängen
  insert_date           date,
  insert_by             varchar(32),
  modified_by           varchar(32),
  modified_date         timestamp(0)
);

-- Indizes
  CREATE INDEX recnokeyword_search
    ON recnokeyword ( r_dbrid, r_kategorie, r_descr );

  CREATE INDEX recnokeyword_r_tablename
    ON recnokeyword ( r_tablename );

  CREATE INDEX recnokeyword_r_kategorie
    ON recnokeyword ( r_kategorie )
        WHERE r_kategorie IS NOT null;

  CREATE INDEX recnokeyword_r_descr
    ON recnokeyword ( r_descr )
        WHERE r_descr IS NOT null;

  CREATE INDEX recnokeyword_r_descr_r_kategorie
    ON recnokeyword ( r_kategorie, r_descr )
        WHERE
                r_kategorie IS NOT null
            AND r_descr IS NOT null;

  CREATE INDEX recnokeyword_r_value_aeoeue_tablename_descr
    ON recnokeyword (
        aeoeue_upper( r_value ) varchar_pattern_ops,
        r_tablename
    )
      WHERE
            aeoeue_upper( r_value ) IS NOT null
        AND r_descr = 'keywordsearch';

  -- Ticket 15150, Index für recnokeyword.r_reg_pname
  CREATE INDEX recnokeyword_r_reg_pname
    ON recnokeyword ( r_reg_pname )
        WHERE r_reg_pname IS NOT null;

  -- Ticket 15702, UQ zur Vermeidung doppelter Druckerzuordnungen
  CREATE UNIQUE INDEX xtt28883 ON recnokeyword( r_dbrid )
  WHERE
          r_kategorie = 'internal system usage'
      AND r_descr     = 'PrinterForReporting';


  -- >> File A 300 System.TDMS.sql -> nachdem custom Typ erstellt wurde
  -- CREATE INDEX ON recnokeyword ( ( ( r_kategorie, r_descr )::tdms.dokument__kategorie__descr  ) );
--

-- Trigger
  CREATE OR REPLACE FUNCTION recnokeyword__b_iu() RETURNS TRIGGER AS $$
    BEGIN
      IF new.r_reg_pname IS NULL THEN -- keine Stammdatenparameter, bezeichnung in dem fall von dort
          IF new.r_descr IS NULL OR new.r_tablename IS NULL OR new.r_dbrid IS NULL THEN -- OR new.r_kategorie IS NULL <= Der Teil macht Probleme,
                                                                                        -- da beim Anlegen von Artikel- oder Projekteigenschaften r_kategorie nicht gefüllt ist.
              RETURN null;
          END IF;
      END IF;

      IF tg_op = 'INSERT' THEN
          IF EXISTS(SELECT true FROM recnokeyword
                    WHERE r_tablename = new.r_tablename
                      AND r_dbrid = new.r_dbrid
                      AND coalesce(r_reg_pname, '') = coalesce(new.r_reg_pname, '')
                      AND coalesce(r_unit, '')      = coalesce(new.r_unit, '')
                      AND coalesce(r_value, '')     = coalesce(new.r_value, '') -- Wenn kein Value da ist, wird der Satz ja scheinbar für was anderes benutzt. Für r_txt als Projekteigenschaft beispielsweise.
                      AND coalesce(r_kategorie, '') = coalesce(new.r_kategorie, '')
                      AND coalesce(r_descr, '')     = coalesce(new.r_descr, '')
                      AND coalesce(r_txt,'')        = coalesce(new.r_txt, '')
                   )
          THEN
              RETURN null; -- doppelte Einträge verwerfen, unterschiedliche r_value mit gleicher Eigenschaft -nicht- (mehr) erlaubt.
              -- (LG: Wieso eigentlich nicht? Und NULL zurückgeben kommt mir langsam generell wie eine schlechte Idee vor.
              --      Wenn es verboten ist, sollte es einen Fehler oder ein Notify geben oder? )
              --  DS: nein, so korrekt. siehe Schlüsselwortsuche
          END IF;
      END IF;
      --
      IF new.r_kategorie IS NULL THEN --keine AND Verknüpfung, diese wird von Postgres immer aufgerufen
          IF new.r_tablename IS NOT NULL AND TSystem.Settings__GetBool('recnokeyword_ForbidFreeInput') AND new.r_tablename NOT IN (
              -- Sind Vorgaben für new.r_desc und entsprechendes Modul angelegt bzw. allgemein Vorgaben (reg_tablename IS dann NULL, also new.r_tablename erlaubt).
              SELECT DISTINCT coalesce(fa_tablename, new.r_tablename) FROM recnogroup LEFT JOIN fieldalias ON upper(fa_fieldname) = upper(reg_tablename) WHERE reg_bez = new.r_descr)
          THEN
              RAISE EXCEPTION '%', lang_text(16215);
          END IF;
      END IF;
      --
      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnokeyword__b_iu
      BEFORE INSERT OR UPDATE
      ON recnokeyword
      FOR EACH ROW
      WHEN  ( TSystem.dms__notify__changes() )  --(current_user NOT IN ('ELO', 'DMS-SYNC-USER', 'syncro'))
      EXECUTE PROCEDURE recnokeyword__b_iu();
  --

  --
  CREATE OR REPLACE FUNCTION recnokeyword__a_iu() RETURNS TRIGGER AS $$
    DECLARE
          picndokudbrid varchar;
          d             varchar;
          str           varchar;
          str1          varchar;
    BEGIN
      --wenn ein Dokument in dem DMS einer Tabelle zugewiesen wird, muß versucht werden aus den Schlüsselbegriffen nun die dbrid der zugehörigen tabelle zuzuweisen
      IF new.r_kategorie IS NOT NULL THEN
          SELECT dbrid
            INTO picndokudbrid
            FROM picndoku
           WHERE dbrid = new.r_dbrid
             AND pd_tablename = new.r_kategorie
             AND coalesce(pd_dbrid, 'DV')='DV';

          IF current_user = 'root' THEN
              RAISE NOTICE 'picndokudbrid:%, new.r_kategorie:%, new.r_descr:%', picndokudbrid, new.r_kategorie, new.r_descr;
          END IF;

          IF picndokudbrid IS NOT NULL THEN
              SELECT rck_column INTO str FROM recnocommentkategorie WHERE rck_id=new.r_kategorie;

              IF (new.r_kategorie || str || quote_literal(new.r_descr)) IS NOT NULL THEN
                  EXECUTE ('SELECT dbrid FROM ' || new.r_kategorie || ' WHERE ' || str || '=' || quote_literal(new.r_descr)) INTO str1;
                  IF str1 IS NOT NULL THEN
                      UPDATE picndoku SET pd_dbrid=str1 WHERE dbrid=picndokudbrid;
                  END IF;
              END IF;
          END IF;
      END IF;
      --
      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnokeyword__a_iu
      AFTER INSERT OR UPDATE
      ON recnokeyword
      FOR EACH ROW
      WHEN (
             ( TSystem.dms__notify__changes() ) --current_user NOT IN ('ELO', 'DMS-SYNC-USER', 'syncro')
          AND new.r_descr IS DISTINCT FROM 'keywordsearch'
      )
      EXECUTE PROCEDURE recnokeyword__a_iu();
  --

  CREATE OR REPLACE FUNCTION recnokeyword__a_iu__dmskeyword() RETURNS TRIGGER AS $$
      DECLARE
          _doktype varchar;
      BEGIN
          -- da Keywords auch vor und rückwärts in andere Dokumente verschlagwortet werden, genau prüfen wo wir sind
          _doktype := pd_doktype FROM picndoku WHERE dbrid = new.r_dbrid;

          -- Schlagwort "Bestellbestätigungsnummer Lieferant" in Bestellung übergeben
          IF new.r_kategorie = 'lds_abnr' THEN
             IF 'bestblief' = _doktype THEN
                 PERFORM teinkauf.ldsdok__ld_abnr__from__dmskeywords__update
                                           (picndoku.pd_dokident::integer,
                                            new.r_descr,
                                            picndoku.pd_date
                                           )
                    FROM picndoku
                   WHERE picndoku.dbrid = new.r_dbrid;
             END IF;
          END IF;

          -- Schlagwort "Angebotsnummer von Lieferant" in Anfrage übergeben
          IF new.r_kategorie = 'anflief' THEN
             IF 'anglief' = _doktype THEN
                 PERFORM teinkauf.anfrage__alief_angNr__from__dmskeywords__update
                                           (picndoku.pd_dokident::integer,
                                            new.r_descr
                                           )
                    FROM picndoku
                   WHERE picndoku.dbrid = new.r_dbrid;
             END IF;
          END IF;

          -- #18779 Aufnahme Dokument am Lieferschein => Schlagwort für den passenden Auftrag hinzugefügt werden
          IF new.r_kategorie = 'belegdokument' THEN

            PERFORM CreateRecNoKeyword( 'auftgtxt', ag_nr, new.r_dbrid )
            FROM auftg WHERE ag_nr IN (
              SELECT DISTINCT ag_nr FROM auftg
              JOIN belegpos ON belp_ag_id = ag_id
              JOIN belegdokument ON beld_id = belp_dokument_id AND beld_belegtyp = 'LFS' AND beld_dokunr = new.r_descr
            );

          END IF;

          RETURN new;
      END $$ LANGUAGE plpgsql;

      CREATE TRIGGER recnokeyword__a_iu__dmskeyword
      AFTER INSERT OR UPDATE
      ON recnokeyword
      FOR EACH ROW
      WHEN (new.r_tablename = 'picndoku')
      EXECUTE PROCEDURE recnokeyword__a_iu__dmskeyword();


  -- Ereignisse bei Artikel-Parametern
  CREATE OR REPLACE FUNCTION recnokeyword__a_iu__art_params() RETURNS TRIGGER AS $$
    DECLARE
          new_value varchar;
          old_value varchar;
    BEGIN
      -- WHEN (new.r_tablename = 'art' AND new.r_reg_pname IS NOT NULL AND new.r_descr IS DISTINCT FROM 'keywordsearch')
      IF new.r_reg_pname = 'art.status.cad' THEN -- IF-THEN-ELSIF: springt bei Treffer nach Ausführung zum Ende (schnellstes)
          IF TG_OP = 'INSERT' THEN
              new_value:= nullif(new.r_value, '');
          ELSE -- UPDATE
              old_value:= nullif(old.r_value, '');
              new_value:= nullif(new.r_value, '');
          END IF;

          IF old_value IS DISTINCT FROM new_value THEN
              INSERT INTO artlog (akl_category    , akl_aknr, akl_cad_event_old, akl_cad_event_new)
              SELECT              'art.status.cad', ak_nr   , old_value        , new_value
              FROM art WHERE dbrid = new.r_dbrid;
          END IF;
      -- anderer Parameter
      -- ELSIF THEN
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnokeyword__a_iu__art_params
      AFTER INSERT OR UPDATE
      OF r_value -- , r_txt -- noch nicht berücksichtigt
      ON recnokeyword
      FOR EACH ROW
      WHEN (
              new.r_tablename = 'art'
          AND new.r_reg_pname IS NOT NULL
          AND new.r_descr IS DISTINCT FROM 'keywordsearch'
      )
      EXECUTE PROCEDURE recnokeyword__a_iu__art_params();
  --

  --
  CREATE OR REPLACE FUNCTION recnokeyword_set_modified() RETURNS TRIGGER AS $$
    DECLARE gruppe2_1 boolean;
    BEGIN
      -- aus Original: siehe recnokeyword_set_modified
      IF tg_op = 'UPDATE' THEN
          IF new.* IS NOT DISTINCT FROM old.* THEN --keine Änderung
              RETURN new;
          END IF;
      END IF;
      --
      IF new.r_tablename = 'qab' THEN --Problemreport
          gruppe2_1 := new.r_reg_pname LIKE 'qab.d%.p%';  --new.r_reg_pname ~* '^qab\\.d3\\.2\\.p\\d+$';

          IF tg_op = 'INSERT' THEN
              IF NOT gruppe2_1 THEN
                  new.insert_by := current_user;
              END IF;

              new.insert_date := current_date;
          END IF;

          IF tg_op = 'UPDATE' THEN
              IF gruppe2_1 THEN
                  IF lower(new.r_value) = 'true' THEN
                      -- wenn insert_by nicht manuel ersetzt wurde
                      IF old.insert_by IS NOT DISTINCT FROM new.insert_by THEN
                          new.insert_by := current_user;
                      END IF;
                  ELSE
                      new.insert_by := NULL;
                  END IF;
              END IF;
          END IF;
      ELSE
          IF tg_op = 'INSERT' THEN
              new.insert_by   := current_user;
              new.insert_date := current_date;
          END IF;
      END IF;

      -- aus Original: siehe recnokeyword_set_modified
      IF NOT modifieddisabled() THEN
          new.modified_by   := current_user;
          new.modified_date := currenttime();
      END IF;

      -- ExternalDMS - Änderungen der Schlagworte   https://redmine.prodat-sql.de/issues/9153 https://redmine.prodat-sql.de/issues/9145
      -- Einschränkungen: z.B. gewisse Keywords nicht als "wichtige" Änderung betrachten und das externe DMS somit nicht benachrichtigen
      --   eventuell entsprechend r_tablename/r_dbrid, r_kategorie/r_descr oder r_reg_pname/r_value
      --   exists(SELECT true FROM information_schema.tables WHERE table_schema = 'public' AND table_name = r_tablename)
      IF
              new.r_tablename = 'picndoku'
          AND NOT modifieddisabled()
          AND EXISTS(SELECT 1 FROM pg_roles WHERE rolname = 'ELO')
      THEN  -- und weitere "Einschränkungen"
          PERFORM TSystem.Settings__Set('DMS__DISABLE__NOTIFY__CHANGES', True);   --- #17067 statt 'ELO' -> Setting: [DMS-SYNC-USER] als DMS-User, denn dort sind die modified-Trigger deaktiviert

          UPDATE picndoku SET
                 pd_external_dms_updated = current_timestamp,
                 pd_external_dms_changed = TSystem.ENUM_SetValue(pd_external_dms_changed, 'keyword')
           WHERE pd_external_dms_id IS NOT NULL
             AND dbrid = new.r_dbrid
             -- es wurde bereits gemarked!
             AND (       pd_external_dms_updated IS DISTINCT FROM current_timestamp
                  OR NOT TSystem.ENUM_GetValue(pd_external_dms_changed, 'keyword')
                 )
          ;

          PERFORM TSystem.Settings__Set('DMS__DISABLE__NOTIFY__CHANGES', False);   --- #17067 statt 'ELO' -> Setting
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

    -- Trigger ÜBERSCHRIEBEN -> heißt wie automatischer DBRID-Trigger "{table}_set_modified" für table_modified()
    CREATE TRIGGER recnokeyword_set_modified
      BEFORE INSERT OR UPDATE
      ON recnokeyword
      FOR EACH ROW
      WHEN ( TSystem.dms__notify__changes() )  -- wie table_modified() zuzüglich DMS-SYNC-USER
      EXECUTE PROCEDURE recnokeyword_set_modified();
  --

-- Sollte in '0 Functions TRecnoParam' benötigt aber die Tabellendefinition
CREATE OR REPLACE FUNCTION TRecnoParam.GetRow(
      pname   varchar,
      rdbrid  varchar
  ) RETURNS recnokeyword AS $$
    SELECT * FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname = pname;
  $$ LANGUAGE sql STABLE;
--

-- Enumerationswerte für Parametertyp ptENUM in recnogroup
CREATE TABLE RecnoEnums (
  rege_id        serial PRIMARY KEY,
  rege_reg_pname varchar(40) NOT NULL, -- Constraint manuell implementiert, da MultiPrimaryKey "reg_pname" durch unterschiedlice AC, welche alle den gleichen reg_pname enthalten können
  rege_code      varchar(30) CHECK (position(',' IN rege_code)=0),  -- Kurzbezeichnung für Anzeige   -- Beispiel: [A+,A,B,C,D,E] -> 6 Datensätze in RecnoEnums
  rege_bez       varchar(100),                                      -- Langbezeichnung               -- Beispiel: 'A+ Großkunden, Eilauftrag'
  rege_pos       integer,

  UNIQUE (rege_reg_pname, rege_code)
);

-- Positionsnummer automatisch hochzählen für neu angelegt Enumwerte
CREATE OR REPLACE FUNCTION recnoEnums__b_i() RETURNS TRIGGER AS $$
  BEGIN
    IF new.rege_pos IS NULL THEN

        new.rege_pos :=
              max( rege_pos )
            FROM recnoEnums
            WHERE rege_reg_pname = new.rege_reg_pname
        ;

        new.rege_pos := coalesce( new.rege_pos, 0 ) + 10;

    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER recnoEnums__b_i
    BEFORE INSERT
    ON recnoenums
    FOR EACH ROW
    EXECUTE PROCEDURE recnoEnums__b_i();
--

-- Grundtabelle für Zubehör der Eigenschaften
-- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/RecnoKeyword_Freie_Felder_EAV
CREATE TABLE recnogroup (
  reg_id                 serial PRIMARY KEY,
  reg_schema             varchar(50),                   -- Schema wo die Tabele aus reg_tablename liegt
  reg_tablename          varchar(80),                   -- (früher reg_tform) Modulname zur Identifikation der F2s
  reg_bez                varchar(100),                  -- Bezeichnung der Artikeleigenschaft Bsp: Oberfläche
  reg_gruppe             varchar(80),                   -- Gruppe zu der sie zugeordnet werden kann
  reg_ac_n               varchar(9),                    -- X TableContraints: REFERENCES artcod ON UPDATE CASCADE; -- Artikel-Code  kann zugeordent werden
  reg_pos                integer,                       -- Position des Parameters innerhalb einer Gruppe für Sortierung in GUI
  reg_pname              varchar(40) NOT NULL,          -- eindeutiger Parametername, damit in Triggern / Funktionen nicht mit ID gearbeitet werden muss.
  reg_gruppe_textno      integer,                       -- Textnr. Übersetzung für Gruppe
  reg_bez_textno         integer,                       -- Textnr. Übersetzung für Bezeichnung
  reg_paramtype          varchar(30),                   -- Parametertypen 'ptVARCHAR', 'ptINTEGER', 'ptNUMERIC', 'ptBOOLEAN', 'ptTRISTATEBOOLEAN', 'ptTIMESTAMP', 'ptDATE', 'ptENUM', 'ptTEXT', 'ptHEX'
  reg_default            varchar(100),                  -- Vorgabe bzw. Standardwert des Parameters (in varchar konvertiert)
  reg_minvalue           integer,                       -- Min-Wert für Zahlen, Mindestlänge bei varchar
  reg_maxvalue           integer,                       -- Max-Wert für Zahlen, Maximallänge bei varchar
  reg_vorgabenliste      boolean DEFAULT FALSE,         -- Vorgabenliste: Enum-Set dient als "Vorgabetabelle" für ein Modul.
  reg_visible            boolean DEFAULT TRUE,          -- Für 'normale' Nutzer in Register Eigenschaften sichtbar
  reg_exclusive          boolean,                       -- Bei Parametertyp ptENUM darf nur ein Enumerationswert angegeben sein, sonst als SET zu betrachten
  reg_autoinsert         boolean NOT NULL DEFAULT FALSE,-- Version 1: Flag in Zieltabellentrigger (z.Bsp. am QAB) abfragen, ob RecnoKeyword automatisch
                                                        -- angelegt werden soll. Version 2: In table_modified verschieben und komplett automatisieren.
  reg_field_name         varchar(50),                   -- Feldname für Vorgabetabelle-Enums, siehe wiki
  reg_field_value        varchar(100),                  -- Wert für Vorgabetabelle-Enums, siehe wiki
  reg_m_iso              varchar(10),                   -- # 7163 EAV - Datentyp Mengeneinheit

  -- CHECK for existing schema
  CONSTRAINT recnogroup__schema__existing CHECK (
          reg_schema IS NULL
      OR  tsystem.schema_exists( reg_schema )
  ),

  -- CHECK for existing table within the current searchpath
  CONSTRAINT recnogroup__table__existing CHECK (
          reg_tablename IS NULL
      OR  tsystem.table_exists( reg_tablename, reg_schema )
  ),

  -- CHECK for existing column
  CONSTRAINT recnogroup__column__existing CHECK (
          reg_field_name IS NULL
      OR  tsystem.column_exists( reg_tablename, reg_field_name, coalesce( reg_schema, 'public' ) )
  ),

  --- #18368 CHECK reg_paramtype in Parametertypen-Liste
  CONSTRAINT reg_paramtype_in_liste CHECK (
          reg_paramtype IN ( 'ptVARCHAR', 'ptINTEGER', 'ptNUMERIC', 'ptBOOLEAN', 'ptTRISTATEBOOLEAN', 'ptTIMESTAMP', 'ptDATE', 'ptENUM', 'ptTEXT', 'ptHEX' )
  )

);

-- Indizes
    CREATE UNIQUE INDEX recnogroup_reg_pname_key ON recnogroup (reg_pname, reg_field_name, reg_field_value);
--

  -- Fkey-CONSTRAINTs auf recnogroup.reg_pname
  CREATE OR REPLACE FUNCTION recnogroup__a_ud__reg_pname__forbid_if_used() RETURNS TRIGGER AS $$
    BEGIN
       -- Änderung von Vorgabe-Eigenschaft, welche in Verwendung sind, prüfen und verbieten
       IF tg_op = 'UPDATE' THEN

          -- Parametername
          IF new.reg_pname IS DISTINCT FROM old.reg_pname THEN

              -- Fkey recnokeyword.r_reg_pname ON UPDATE RESTRICT
              IF EXISTS(SELECT true FROM recnokeyword WHERE r_reg_pname = old.reg_pname) THEN
                  -- Umschreiben bzw. Löschen des Parameters nicht möglich, da in Verwendung.
                  RAISE EXCEPTION 'parameter_change_denied xtt16571';
              END IF;

              -- Fkey RecnoEnums.rege_reg_pname ON UPDATE CASCADE
              UPDATE RecnoEnums SET rege_reg_pname = new.reg_pname WHERE rege_reg_pname = old.reg_pname;

          END IF;

          -- Parametertyp
          IF new.reg_paramtype IS DISTINCT FROM old.reg_paramtype THEN

              -- recnogroup.reg_paramtype ON UPDATE RESTRICT
              IF EXISTS(SELECT true FROM recnoenums WHERE rege_reg_pname = old.reg_pname) THEN
                  -- Umschreiben des ENUM-Typs nicht möglich, bis ENUM-Liste nicht leer ist.
                  RAISE EXCEPTION 'enum-type_change_denied xtt16625';
              END IF;

          END IF;

       END IF;

       -- Löschen von Vorgabe-Eigenschaft
       IF tg_op = 'DELETE' THEN

          -- Fkey recnokeyword.r_reg_pname ON DELETE RESTRICT
          IF EXISTS(SELECT true FROM recnokeyword WHERE r_reg_pname = old.reg_pname) THEN
              -- Umschreiben bzw. Löschen des Parameters nicht möglich, da in Verwendung.
              RAISE EXCEPTION 'parameter_change_denied xtt16571';
          END IF;

          -- Fkey RecnoEnums.rege_reg_pname ON DELETE CASCADE
          DELETE FROM RecnoEnums WHERE rege_reg_pname = old.reg_pname;

          RETURN old;

       END IF;


       RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnogroup__a_ud__reg_pname__forbid_if_used
      AFTER DELETE OR UPDATE
      OF reg_pname, reg_paramtype
      ON recnogroup
      FOR EACH ROW
      EXECUTE PROCEDURE recnogroup__a_ud__reg_pname__forbid_if_used();
  --

  -- Positionsnummer automatisch ab 10 in 10-Schritte hochzählen für neu angelegten Parameter
  -- Vorgabelliste und ENUM-Typ
  CREATE OR REPLACE FUNCTION recnogroup__b_i() RETURNS TRIGGER AS $$
    BEGIN
      -- Keine Positionsnummer aus Oberfläche
      IF new.reg_pos IS NULL THEN
          new.reg_pos :=
                max( reg_pos )
              FROM recnogroup
              WHERE
                    -- innerhalb einer Gruppe
                      coalesce(     reg_gruppe, lang_text(     reg_gruppe_textno ), lang_text( 10084 ) )
                    = coalesce( new.reg_gruppe, lang_text( new.reg_gruppe_textno ), lang_text( 10084 ) )
          ;

          new.reg_pos := coalesce (new.reg_pos, 0 ) + 10;

      END IF;

      -- Bei Vorgabeliste auf ENUM-Typ setzen
      IF new.reg_vorgabenliste THEN
          IF new.reg_paramtype IS NULL THEN
              new.reg_paramtype := 'ptENUM';
          END IF;
      END IF;

      -- Vorgabenliste nur beim ENUM-Typ möglich
      IF new.reg_paramtype <> 'ptENUM' THEN
          new.reg_vorgabenliste := false;
      END IF;

      -- Parametername automatisch aus Tablename und Fieldname generieren
      IF new.reg_pname IS NULL THEN

          new.reg_pname := new.reg_tablename;

          IF new.reg_field_name IS NOT NULL THEN
              new.reg_pname := new.reg_pname || '.' || new.reg_field_name;
          END IF;

      END IF;


      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnogroup__b_i
      BEFORE INSERT
      ON recnogroup
      FOR EACH ROW
      EXECUTE PROCEDURE recnogroup__b_i();
  --
    CREATE OR REPLACE FUNCTION recnogroup__b_iu_check_umlaute() RETURNS TRIGGER AS $$
      BEGIN
        -- Namen nur korrigieren wenn es noch keine Werte in der recnokeyword gibt
        IF NOT EXISTS( SELECT TRUE FROM recnokeyword WHERE r_reg_pname = new.reg_pname) THEN
          NEW.reg_pname := regexp_replace(NEW.reg_pname, 'ä', 'ae', 'gi');
          NEW.reg_pname := regexp_replace(NEW.reg_pname, 'ö', 'oe', 'gi');
          NEW.reg_pname := regexp_replace(NEW.reg_pname, 'ü', 'ue', 'gi');
          NEW.reg_pname := regexp_replace(NEW.reg_pname, 'ß', 'ss', 'gi');
        END IF;

        RETURN NEW;
      END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnogroup__b_iu_check_umlaute
      BEFORE INSERT OR UPDATE
      ON recnogroup
      FOR EACH ROW
      EXECUTE FUNCTION recnogroup__b_iu_check_umlaute();

  -- Default - Werte
  CREATE OR REPLACE FUNCTION recnogroup__b_iu_defaults() RETURNS TRIGGER AS $$
    BEGIN
      -- Vorgaben aus RecNoGroup prüfen

      -- varchar
      IF new.reg_paramtype='ptVARCHAR' THEN
          IF new.reg_minvalue IS NOT NULL THEN
              IF isInteger(new.reg_minvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF length(new.reg_default) < new.reg_minvalue::integer THEN
                          RAISE EXCEPTION '%', Format(lang_text(29157) /*'reg_default muss mehr als % Buchstaben erhalten!'*/, (new.reg_minvalue-1));
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29161) /*'reg_minvalue muss integer sein!'*/);
              END IF;
          END IF;

          IF new.reg_maxvalue IS NOT NULL THEN
              IF isInteger(new.reg_maxvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF length(new.reg_default) > new.reg_maxvalue::integer THEN
                          RAISE EXCEPTION '%', Format(lang_text(29157) /*'reg_default muss weniger als % Buchstaben erhalten!'*/, (new.reg_maxvalue+1));
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29158) /*'reg_maxvalue muss integer sein!'*/);
              END IF;
          END IF;

      -- integer und Hex
      ELSEIF new.reg_paramtype in ('ptINTEGER', 'ptHEX') THEN
          IF new.reg_minvalue IS NOT NULL THEN
              IF isInteger(new.reg_minvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF isInteger(new.reg_default) THEN
                          IF new.reg_default::integer < new.reg_minvalue::integer THEN
                              RAISE EXCEPTION '%', Format(lang_text(29159) /*'reg_default muss größer oder gleich reg_minvalue sein!'*/);
                          END IF;
                      ELSE
                          RAISE EXCEPTION '%', Format(lang_text(29160) /*'reg_default muss integer sein!'*/);
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29161) /*'reg_minvalue muss integer sein!'*/);
              END IF;
          END IF;

          IF new.reg_maxvalue IS NOT NULL THEN
              IF isInteger(new.reg_maxvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF isInteger(new.reg_default) THEN
                          IF new.reg_default::integer > new.reg_maxvalue::integer THEN
                              RAISE EXCEPTION '%', Format(lang_text(29262) /*'reg_default muss kleiner oder gleich reg_maxvalue sein!'*/);
                          END IF;
                      ELSE
                          RAISE EXCEPTION '%', Format(lang_text(29160) /*'reg_default muss integer sein!'*/);
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29158) /*'reg_maxvalue muss integer sein!'*/);
              END IF;
          END IF;

      --  numeric
      ELSEIF new.reg_paramtype='ptNUMERIC' THEN
          IF new.reg_minvalue IS NOT NULL THEN
              IF isNumeric(new.reg_minvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF isNumeric(new.reg_default) THEN
                          IF new.reg_default::numeric < new.reg_minvalue::numeric THEN
                              RAISE EXCEPTION '%', Format(lang_text(29159) /*'reg_default muss größer oder gleich reg_minvalue sein!'*/);
                          END IF;
                      ELSE
                          RAISE EXCEPTION '%', Format(lang_text(29163) /*'reg_default muss numeric sein!'*/);
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29164) /*'reg_minvalue muss numeric sein!'*/);
              END IF;
          END IF;

          IF new.reg_maxvalue IS NOT NULL THEN
              IF isNumeric(new.reg_maxvalue) THEN
                  IF new.reg_default IS NOT NULL THEN
                      IF isNumeric(new.reg_default) THEN
                          IF new.reg_default::numeric > new.reg_maxvalue::numeric THEN
                              RAISE EXCEPTION '%', Format(lang_text(29262) /*'reg_default muss kleiner oder gleich reg_maxvalue sein!'*/);
                          END IF;
                      ELSE
                          RAISE EXCEPTION '%', Format(lang_text(29163) /*'reg_default muss numeric sein!'*/);
                      END IF;
                  END IF;
              ELSE
                  RAISE EXCEPTION '%', Format(lang_text(29165) /*'reg_maxvalue muss numeric sein!'*/);
              END IF;
          END IF;

      --  boolean und 3-STATE-boolean
      ELSEIF new.reg_paramtype in ('ptBOOLEAN', 'ptTRISTATEBOOLEAN') THEN
          -- min/max ohne Prüfung
          IF new.reg_default IS NOT NULL THEN
              IF NOT(isBoolean(new.reg_default)) THEN
                  RAISE EXCEPTION '%', Format(lang_text(29166) /*'reg_default muss boolean sein!'*/);
              END IF;
          END IF;
      END IF;

      RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER recnogroup__b_iu_defaults
      BEFORE INSERT OR UPDATE
      ON recnogroup
      FOR EACH ROW
      EXECUTE PROCEDURE recnogroup__b_iu_defaults();
--

-- Tabelle in der Bemerkungen/Wiedervorlagen zu jedem Datensatz hinterlegt sein können.
CREATE TABLE recnocomments(
  rc_id                 serial PRIMARY KEY,
  rc_tablename          varchar(50),
  rc_dbrid              varchar(32) NOT NULL,
  rc_parent_dbrid       varchar(32),
  rc_user               varchar(30) DEFAULT tsystem.current_user_ll_db_usename(TRUE), -- REFERENCES llv (ll_db_usename) ON UPDATE CASCADE;
  rc_date               timestamp(0)  DEFAULT currenttime(),
  rc_betreff            varchar(100) NOT NULL,
  rc_text               text,
  rc_text_rtf           text,
  rc_category           varchar(30),
  rc_print              bool NOT NULL DEFAULT true,
  rc_allg1              varchar(150), -- Modulüberschrift usw beispielsweise bei WVOD
  rc_wvod               date,
  rc_wvod_params        text
);

-- Indizes
    CREATE INDEX recnocomments_main ON recnocomments(rc_tablename, rc_dbrid);
    CREATE INDEX recnocomments_rdbrid ON recnocomments(rc_dbrid);
    CREATE INDEX recnocomments_rparent_dbrid ON recnocomments(rc_parent_dbrid) WHERE rc_parent_dbrid IS NOT null;
    -- ACHTUNG: die Indizes sind genau auf FUNCTION get_record_note_id abgestimmt!
    CREATE INDEX recnocomments_rdbrid__get_record_note_id ON recnocomments(rc_dbrid, rc_date) INCLUDE (rc_id) WHERE rc_wvod IS null;
    CREATE INDEX recnocomments_rparent_dbrid__get_record_note_id ON recnocomments(rc_parent_dbrid, rc_date) INCLUDE (rc_id) WHERE rc_parent_dbrid IS NOT null AND rc_wvod IS null;
--



  -- Gibt den Vorgabe-Standardwert als varchar zurück.
  CREATE OR REPLACE FUNCTION TRecnoParam.recnogroup__default_get(
        _tablename              varchar,
        _pname                  varchar,
        _ErrorIfDefaultIsEmpty  boolean = false
  ) RETURNS text AS $$
  DECLARE
    defValue  text;
    pType     varchar;
  BEGIN
    IF coalesce(_pname, '') = '' THEN
      RAISE EXCEPTION 'TRecnoParam.recnogroup__default_get: Missing IN-Parameter _pname';
    END IF;


    SELECT
      reg_default
    INTO
      defValue
    FROM
      recnogroup
    WHERE
          reg_tablename = _tablename
      AND reg_pname     = _pname;

    IF _ErrorIfDefaultIsEmpty
      AND coalesce(defValue, '') = ''
    THEN
      RAISE EXCEPTION 'TRecnoParam.recnogroup__default_get: Missing Default-Value for Parameter %.', _pname;
    END IF;

    RETURN defValue;
  END $$ LANGUAGE plpgsql;
--


/********************** R E C N O K E Y W O R D   -   P A R A M E T E R ******************/
/*
  Verfügbare Parametertypen u. Funktionen, ermittelbar mit SELECT TRecnoParam.SupportedTypes();
    "(ptBOOLEAN,   boolean)"      Get/Set,               Konvertierung: Manuell
    "(ptDATE,      Date)"         Get/Set,               Konvertierung: Cast, Format unklar
    "(ptENUM,      Enumeration)"  Get/Set als Varchar    Konvertierung: Kein Cast als Varchar      Zusatzfunktionen für Enum-Flags
    "(ptINTEGER,   integer)"      Get/Set,               Konvertierung: Cast
    "(ptNUMERIC,   Numeric)"      Get/Set,               Konvertierung: Cast, Format unklar
    "(ptTIMESTAMP, TimeStamp)"    Get/Set,               Konvertierung: Cast, Format unklar
    "(ptVARCHAR,   Varchar)"      Get/Set,               Konvertierung: Cast
    "(ptText,      Text)"         Get/Set,               Konvertierung: Kein Cast, in r_txt als Text
    "(ptHEX,       Integer)"      Get/Set als Integer,   Konvertierung: Cast

  Neue Parameter definieren : Stammdaten > Vorgaben > Zubehör > Eigenschaften / Parameter

  Mindestens anzugeben: Bezeichnung, Typ und Parametername
  Neue Tabelle: einfach irgendwo einen neuen Parameter einfügen und "Tabellenname" ändern (danach Fenster neu öffnen, damit Gruppe erscheint)

  Undefinierte Parameter werden stillschweigend verworfen, wenn man versucht sie zu setzen. (TRecnoParam.Set)


  TRecnoParam.SupportedTypes   (OUT pt_typ VARCHAR, OUT pt_bez VARCHAR) RETURNS SETOF record
  TRecnoParam.GetBez           (pname VARCHAR) RETURNS VARCHAR

  TRecnoParam.GetBool          (pname VARCHAR, rdbrid VARCHAR, defaultValue boolean      DEFAULT NULL) RETURNS boolean
  TRecnoParam.GetDate          (pname VARCHAR, rdbrid VARCHAR, defaultValue DATE         DEFAULT NULL) RETURNS DATE
  TRecnoParam.GetEnum          (pname VARCHAR, rdbrid VARCHAR, defaultValue VARCHAR(100) DEFAULT NULL) RETURNS VARCHAR(100)
  TRecnoParam.GetInteger       (pname VARCHAR, rdbrid VARCHAR, defaultValue integer      DEFAULT NULL) RETURNS integer
  TRecnoParam.GetNumeric       (pname VARCHAR, rdbrid VARCHAR, defaultValue NUMERIC      DEFAULT NULL) RETURNS NUMERIC
  TRecnoParam.GetTimeStamp     (pname VARCHAR, rdbrid VARCHAR, defaultValue TIMESTAMP    DEFAULT NULL) RETURNS TIMESTAMP
  TRecnoParam.GetVarchar       (pname VARCHAR, rdbrid VARCHAR, defaultValue VARCHAR(100) DEFAULT NULL) RETURNS VARCHAR(100)
  TRecnoParam.GetText          (pname VARCHAR, rdbrid VARCHAR, defaultValue TEXT         DEFAULT NULL) RETURNS TEXT
  --TRecnoParam.GetValue       (pname VARCHAR, rdbrid VARCHAR, defaultValue TEXT         DEFAULT NULL) RETURNS TEXT  -- BasisFunktion

  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue boolean)
  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue DATE)
  --TRecnoParam.Set            (pname VARCHAR, rdbrid VARCHAR, rValue VARCHAR(100))      -- ENUM wird von VARCHAR behandelt
  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue integer)
  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue NUMERIC)
  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue TIMESTAMP)
  TRecnoParam.Set              (pname VARCHAR, rdbrid VARCHAR, rValue VARCHAR(100))
  TRecnoParam.SetText          (pname VARCHAR, rdbrid VARCHAR, rvalue TEXT, AllowUpdate boolean DEFAULT TRUE)
  --TRecnoParam.SetValue       (pname VARCHAR, rdbrid VARCHAR, rvalue TEXT, AllowUpdate boolean DEFAULT TRUE)  -- BasisFunktion

  TRecnoParam.SetDefault       (pname VARCHAR, rdbrid VARCHAR, AllowUpdate boolean DEFAULT TRUE, ErrorIfDefaultIsEmpty boolean DEFAULT FALSE) RETURNS TEXT
  TRecnoParam.Exists           (pname VARCHAR, rdbrid VARCHAR) RETURNS boolean
  TRecnoParam.Delete           (pname VARCHAR, rdbrid VARCHAR) RETURNS void
  --TRecnoParam.GetRow         (pname VARCHAR, rdbrid VARCHAR) RETURNS recnokeyword    -- musste nach A_system.sql verschoben werden

  TRecnoParam.Enum_ContainsFlag(pname VARCHAR, rdbrid VARCHAR, regeCode VARCHAR) RETURNS boolean
  TRecnoParam.Enum_AddFlag     (pname VARCHAR, rdbrid VARCHAR, regeCode VARCHAR)
  TRecnoParam.Enum_RemoveFlag  (pname VARCHAR, rdbrid VARCHAR, regeCode VARCHAR)
  TRecnoParam.Enum_FlipFlag    (pname VARCHAR, rdbrid VARCHAR, regeCode VARCHAR) RETURNS boolean

  TRecnoParam.Copy             (pName     VARCHAR, srcDBRID VARCHAR, trgDBRID VARCHAR) RETURNS boolean
  TRecnoParam.CopyGroup        (groupName VARCHAR, srcDBRID VARCHAR, trgDBRID VARCHAR) RETURNS integer
  TRecnoParam.CopyAll          (                   srcDBRID VARCHAR, trgDBRID VARCHAR) RETURNS integer


  alte/weitere Funktionen:
    RecNoParams ohne PName und ohne nötige Registrierung in "Eigenschaften / Parameter"
    siehe set_recnoparam und get_recnoparam (mit rtablename, rdbrid und rdescr)

    0_Functions_TSystem.sql
      SetPrintSetting        (tablename VARCHAR, rvalue VARCHAR, rdbrid VARCHAR)
      TSystem.SetPrintSetting(tablename VARCHAR, rvalue VARCHAR, rdbrid VARCHAR, remove boolean DEFAULT false)
      TSystem.Get_PrintSettings(INOUT ps_dbrid VARCHAR, OUT ps_settings VARCHAR) RETURNS record
      TSystem.CreateKeyword(rkategorie VARCHAR, rtablename VARCHAR, rdescr VARCHAR, rdbrid VARCHAR, rvalue VARCHAR, pname VARCHAR, runit VARCHAR DEFAULT NULL)
      TSystem.DeleteKeyword(rkategorie VARCHAR,                                     rdbrid VARCHAR, rdescr VARCHAR, pname VARCHAR DEFAULT NULL)

    0_Functions.sql
      set_recnoparam(rtablename VARCHAR, rdbrid VARCHAR,       rdescr VARCHAR,     rvalue VARCHAR,     rnot boolean DEFAULT FALSE)
      get_recnoparam(rtablename VARCHAR, rdbrid VARCHAR, INOUT rdescr VARCHAR, OUT rvalue VARCHAR, OUT rnot boolean, OUT rkategorie VARCHAR, OUT rtxt TEXT, OUT modby VARCHAR) RETURNS record
      del_recnoparam(rtablename VARCHAR, rdbrid VARCHAR,       rdescr VARCHAR)

    A_system.sql
      CreateRecNoKeyword(tablename VARCHAR, value integer, ddbrid VARCHAR)
      CreateRecNoKeyword(tablename VARCHAR, value VARCHAR, ddbrid VARCHAR)
      DeleteRecNoKeyword(tablename VARCHAR, value integer, ddbrid VARCHAR DEFAULT NULL)
      CheckRecNoKeyword (tablename VARCHAR, value integer, ddbrid VARCHAR DEFAULT NULL) RETURNS bool
      CheckRecNoKeyword (tablename VARCHAR, value VARCHAR, ddbrid VARCHAR DEFAULT NULL) RETURNS bool
      --set_recnoparam() RETURNS TRIGGER
      TRecnoParam.GetRow(pname VARCHAR, rdbrid VARCHAR) RETURNS recnokeyword
      isParamRight(val VARCHAR, pname VARCHAR) RETURNS boolean

      hier drin sind auch die RecNoKeyword-Tabellen und Picndoku-Trigger (basierend auf CreateRecNoKeyword und CreatePicnDokuLink)

      CreatePicnDokuLink(pdid integer, tablename VARCHAR, tdbrid VARCHAR)
      DeletePicnDokuLink(pdid integer, tablename VARCHAR, tdbrid VARCHAR)
      MovePicnDokuLink(pdid integer, tablename VARCHAR, tdbrid_old VARCHAR, tdbrid_new VARCHAR)

    XX_Functions_TSystem_Recno.sql
      TSystem.kws_generate_split_keywords            (inString VARCHAR, return_orig boolean DEFAULT FALSE, OUT keyword VARCHAR) RETURNS SETOF VARCHAR
      TSystem.kws_generate_split_keywords_full_string(inString VARCHAR) RETURNS VARCHAR
      TSystem.kws_get_recnodbrids_by_keyword_and     (inSearchString VARCHAR, inSearchFields VARCHAR, trg_table VARCHAR, strictSearch boolean DEFAULT FALSE, OUT dbrid VARCHAR, OUT hitfields VARCHAR) RETURNS SETOF record
      TSystem.kws_get_recnodbrids_by_keyword_mask    (inSearchString VARCHAR, inSearchFields VARCHAR, trg_table VARCHAR, strictSearch boolean DEFAULT FALSE, OUT dbrid VARCHAR, OUT bitmask integer, OUT hitfields VARCHAR) RETURNS SETOF record
      --TSystem.kws_CreateKeyword (kategorie VARCHAR, tablename VARCHAR, dbrid VARCHAR, value VARCHAR, pname VARCHAR, unit VARCHAR)
      tsystem.kws_createkeyword   (tablename VARCHAR, value VARCHAR, unit VARCHAR, ddbrid VARCHAR, descr VARCHAR DEFAULT 'keywordsearch'::VARCHAR)
      TSystem.kws_CreateKeyword   (tablename VARCHAR, value integer, unit VARCHAR, ddbrid VARCHAR)
      TSystem.kws_create_keywords (dataRow ADK)
      TSystem.kws_create_keywords (dataRow ART)
      TSystem.kws_create_keywords (dataRow AUFTG)
      TSystem.kws_create_keywords (dataRow LDSDOK)

    PrintSettings.pas
      TPrintSettings
        SetPrintSetting           // SET zuweisen
        GetPrintSettingsCount     // Anzahl der möglichen ENUM/SET-Werte (TPrintSettings/TPrintSettingsSet)
        PrintSettingsToDBString   // ENUM/SET zu KommaString
        DBStringToPrintSettings   // KommaString zu ENUM/SET

      TPrintSettings baut ", (TReporting.recnokeyword__print_setting__get(xxx.dbrid)).*" vor das FROM in SqlSelect der angehangenen CimClass2 ein
      Spalten ps_dbrid und ps_settings (der KommaString)
*/

-- Gibt mögliche Parameter-Typen zurück. Unter anderem verwendet als Listsource für Auswahl des Typs im cxGrid.
-- TODO: Textnummern / Übersetzung für Parametertypen. Die werden dem Nutzer unter Umständen angezeigt.
-- ToDo: Enum als varchar pt_dbtyp ist noch nicht korrekt
CREATE OR REPLACE FUNCTION TRecnoParam.SupportedTypes(OUT pt_typ varchar, OUT pt_bez varchar, OUT pt_dbtyp varchar) RETURNS SETOF record AS $$
 DECLARE rec record;
 BEGIN
  FOR rec IN (SELECT 'ptVARCHAR' AS pt_typ, 'String (Kurz)' AS pt_bez, 'varchar' AS pt_dbtyp
              UNION
              SELECT 'ptINTEGER'          , 'Integer'                , 'integer'
              UNION
              SELECT 'ptNUMERIC'          , 'Numeric'                , 'numeric'
              UNION
              SELECT 'ptBOOLEAN'          , 'Boolean (2-State)'      , 'boolean'
              UNION
              SELECT 'ptTRISTATEBOOLEAN'  , 'Boolean (3-State)'      , 'boolean'
              UNION
              SELECT 'ptTIMESTAMP'        , 'TimeStamp'              , 'timestamp'
              UNION
              SELECT 'ptDATE'             , 'Date'                   , 'date'
              UNION
              SELECT 'ptENUM'             , 'Enumeration'            , 'varchar' --todo
              UNION
              SELECT 'ptTEXT'             , 'Text (Unbegrenzt)'      , 'text'
              UNION
              SELECT 'ptHEX'              , 'Hexadezimalzahl'        , 'integer'
              ORDER BY pt_bez) LOOP
    pt_typ:=rec.pt_typ;
    pt_bez:=rec.pt_bez;
    pt_dbtyp:=rec.pt_dbtyp;
    RETURN NEXT;
 END LOOP;
 RETURN;
 END $$ LANGUAGE plpgsql IMMUTABLE;
--


-- [TRIGGERFUNKTION] TRecnoParam.CreateAutoParams: Kann vom After-Insert Trigger an beliebigen Tabellen aufgerufen werden.
-- Legt die Vorgabeparameter zur Tabelle mit Default-Werten an.
CREATE OR REPLACE FUNCTION TRecnoParam.CreateAutoParams() RETURNS TRIGGER AS $$
  DECLARE
      _rdbrid           varchar;
      _value            varchar;
      _recnogroup_auto  record;
  BEGIN
    -- Lieferantenprüfungen: Wenn adk2-Insert, dann adk-Param durch adk2__a_i__create_autoparams
    IF TG_TABLE_NAME = 'adk2' THEN

        _rdbrid :=
              adk.dbrid
            FROM adk
              JOIN adk2 ON ad_krz = a2_krz
            WHERE adk2.dbrid = new.dbrid
        ;

        PERFORM TRecnoParam.SetDefault('Lieferant.Geheimhaltungsvereinbarung', _rdbrid, false);
        PERFORM TRecnoParam.SetDefault('Lieferant.Selbstauskunft',             _rdbrid, false);

    END IF;


    -- Parameter mit Auto-Insert Kennzeichen durchgehen, die zur aktuellen Tabelle zugehörig sind.
    FOR _recnogroup_auto IN
        SELECT
          reg_field_name,
          reg_field_value,
          reg_pname
        FROM recnogroup
        WHERE reg_autoinsert
          AND reg_tablename = TG_TABLE_NAME
    LOOP
        -- anschliessend prüfen, ob der Parameter einen Feldwert/Value hat.
        IF coalesce( _recnogroup_auto.reg_field_name, '' ) <> '' THEN

            -- Wert vom Feld aus reg_field_name aus der Tabelle TG_TABLE_NAME holen.
            _value := to_jsonb( new )->>_recnogroup_auto.reg_field_name;

            -- Wenn Feld und Value vorhanden, prüfen wir ob Feld und Value der TG_TABLE gleich sind.
            -- wie in recnogroup definiertund legen nur dann entsprechenden Parameter an.
            IF _recnogroup_auto.reg_field_value IS NOT DISTINCT FROM _value THEN -- IN value list

                PERFORM
                    TRecnoParam.SetDefault(
                        _recnogroup_auto.reg_pname,
                        new.dbrid,
                        false,
                        false,
                        _recnogroup_auto.reg_field_name,
                        _recnogroup_auto.reg_field_value
                    )
                ;

            END IF;

        -- Wenn nein, fügen wir die Parameter immer ein, sie haben keine Einschränkung.
        ELSE

            PERFORM
                TRecnoParam.SetDefault(
                    _recnogroup_auto.reg_pname,
                    new.dbrid,
                    false
                )
            ;

        END IF;

    END LOOP;


    RETURN new;
  END $$ LANGUAGE plpgsql;
--


-- EAV FUNKTIONEN

-- WERTE LADEN / Anzeigen / Holen. zB Alle Parameter am Datensatz, einen bestimmten Parameter usw.

  -- Bezeichnung des Parameters
  CREATE OR REPLACE FUNCTION TRecnoParam.GetBez(IN pname varchar) RETURNS varchar AS $$
    BEGIN
      RETURN (SELECT coalesce(reg_bez, lang_text(reg_bez_textno)) FROM RecNoGroup WHERE reg_pname=pname);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Boolean-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetBool(IN pname varchar, IN rdbrid varchar, IN defaultValue boolean DEFAULT NULL) RETURNS boolean AS $$
    BEGIN
      RETURN coalesce( CAST( TRecnoParam.GetValue(pname, rdbrid) AS boolean), defaultVALUE);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Date-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetDate(IN pname varchar, IN rdbrid varchar, IN defaultValue DATE DEFAULT NULL) RETURNS DATE AS $$
    BEGIN
      RETURN coalesce( CAST( TRecnoParam.GetValue(pname, rdbrid) AS DATE), defaultVALUE);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Enum-Parameter) - Gibt einfach den String zurück.
  -- Dort stehen kommasepariert die Codes der Enum-Werte drin (definiert in recnoEnums.rege_code)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetEnum(IN pname varchar, IN rdbrid varchar, IN defaultValue varchar(100) DEFAULT NULL) RETURNS varchar(100) AS $$
    BEGIN
      RETURN CAST(coalesce( TRecnoParam.GetValue(pname, rdbrid), defaultVALUE) AS varchar(100));
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Integer-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetInteger(IN pname varchar, IN rdbrid varchar, IN defaultValue integer DEFAULT NULL) RETURNS integer AS $$
    BEGIN
      RETURN coalesce( CAST( TRecnoParam.GetValue(pname, rdbrid) AS integer), defaultVALUE);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Numeric-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetNumeric(IN pname varchar, IN rdbrid varchar, IN defaultValue numeric DEFAULT NULL) RETURNS numeric AS $$
    BEGIN
      RETURN coalesce( CAST( TRecnoParam.GetValue(pname, rdbrid) AS numeric), defaultVALUE);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (TimeStamp-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetTimeStamp(IN pname varchar, IN rdbrid varchar, IN defaultValue TIMESTAMP WITHOUT TIME ZONE DEFAULT NULL) RETURNS TIMESTAMP WITHOUT TIME ZONE AS $$
    BEGIN
      RETURN coalesce( CAST( TRecnoParam.GetValue(pname, rdbrid) AS TIMESTAMP WITHOUT TIME ZONE ), defaultVALUE);
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (varchar/String Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetVarchar(IN pname varchar, IN rdbrid varchar, IN defaultValue varchar(100) DEFAULT NULL) RETURNS varchar(100) AS $$
    BEGIN
      RETURN CAST( coalesce(TRecnoParam.GetValue(pname, rdbrid, defaultValue),defaultValue) AS varchar(100));
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- [BASISFUNKTION] Zur Suche nach einem Recnokeyword-Parameter an einem bestimmten Datensatz.
  -- DefaultValue wird zurückgegeben, wenn in der DB kein Parameter gefunden wurde.
  CREATE OR REPLACE FUNCTION TRecnoParam.GetValue(IN pname varchar, IN rdbrid varchar, IN defaultValue text DEFAULT NULL) RETURNS text AS $$
    DECLARE
        val text;
    BEGIN
      -- Das knallt, wenn es mehr als einen Datensatz gibt, was aber OK ist, da wir implizit überall annehmen das jeder Parameter nur einmal pro Zieldatensatz auftauchen darf.
      SELECT r_value INTO val FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname = pname;
      -- Wenn nichts gefunden wurde, nehmen wir die Vorgabe
      IF coalesce(val,'') = '' THEN
          val := defaultValue;
      END IF;

      RETURN val;
    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Text Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.GetText(IN pname varchar, IN rdbrid varchar, IN defaultValue text DEFAULT NULL) RETURNS text AS $$
   DECLARE
        val text;
   BEGIN
     -- Das knallt, wenn es mehr als einen Datensatz gibt, was aber OK ist, da wir implizit überall annehmen das jeder Parameter nur einmal pro Zieldatensatz auftauchen darf.
     SELECT r_txt INTO val FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname = pname;
     -- Wenn nichts gefunden wurde, nehmen wir die Vorgabe
     IF coalesce(val,'') = '' THEN
        val := defaultValue;
     END IF;

     RETURN val;
   END $$ LANGUAGE plpgsql STABLE;
  --

  -- EAV Parameter pro Record als Einzelzeile. plsql Funktion, da art noch nicht existiert.
  CREATE OR REPLACE FUNCTION TRecnoParam.get__all_as_rows__by__tablename_dbrid(
        _tname                                varchar,
        _dbrid                                varchar,
        OUT pname                             varchar,
        OUT ptxtnr                            integer,
        OUT r_descr__reg_bez                  varchar,
        OUT r_descr                           varchar,
        OUT reg_bez                           varchar,
        OUT r_descr__reg_bez__as__fieldalias  varchar,
        OUT r_value                           varchar,
        OUT reg_gruppe                        varchar
    ) RETURNS SETOF record AS $$
      SELECT
        r_reg_pname,
        reg_bez_textno,
        coalesce(r_descr, reg_bez) AS r_descr__reg_bez,
        reg_bez,
        r_descr,
        getFieldAlias( coalesce(r_descr, reg_bez) ) AS r_descr__reg_bez__as__fieldalias,
        r_value,
        reg_gruppe
      FROM RecNoKeyword
        LEFT JOIN recnogroup ON reg_pname = r_reg_pname
      WHERE r_tablename = _tname
        AND r_dbrid = _dbrid
        AND coalesce(r_kategorie, '') NOT IN ('internal system usage')
        AND coalesce(reg_visible, TRUE)
      ORDER BY reg_pos
      ;

    $$ LANGUAGE sql;
  --

-- [BASISFUNKTION] Prüft ob der Parameter 'pname' am Datensatz 'rdbrid' gesetzt wurde, also in Recnokeyword vorhanden ist.
  CREATE OR REPLACE FUNCTION TRecnoParam.Exists(IN pname varchar, IN rdbrid varchar) RETURNS boolean
    AS $$
      SELECT EXISTS(SELECT true FROM recnokeyword WHERE r_dbrid = rdbrid AND r_reg_pname = pname);
    $$ LANGUAGE sql STABLE STRICT;
  --

  -- Prüft ob zu einer Gruppe und eines Tables mindestens ein Parameter gesetzt ist #19747
  CREATE OR REPLACE FUNCTION TRecnoParam.Exists__by__tablename_group(
      _tname varchar,
      _group varchar
  ) RETURNS boolean AS $$
    DECLARE
      _query TEXT;
      _result boolean;
    BEGIN
        _query := 'SELECT EXISTS (SELECT true FROM ' || _tname ||
                  ' JOIN LATERAL (SELECT pname, reg_gruppe FROM TRecnoParam.get__all_as_rows__by__tablename_dbrid(' || quote_literal(_tname) || ', ' || _tname || '.dbrid)) AS params ON params.reg_gruppe = ' || quote_literal(_group) ||
                  ' LIMIT 1)';

        EXECUTE _query INTO _result;
        RETURN _result;

    END $$ LANGUAGE plpgsql STABLE STRICT;

  --
  CREATE OR REPLACE FUNCTION TRecnoParam.get__all_as_rows_and_predecessor__by__tablename_dbrid(
        _tname  varchar,
        _dbrid  varchar,

        -- vgl. TRecnoParam.get__all_as_rows__by__tablename_dbrid
        OUT pname                             varchar,
        OUT ptxtnr                            integer,
        OUT r_descr__reg_bez                  varchar,
        OUT r_descr                           varchar,
        OUT reg_bez                           varchar,
        OUT r_descr__reg_bez__as__fieldalias  varchar,
        OUT r_value                           varchar,
        OUT reg_gruppe                        varchar
  ) RETURNS SETOF record AS $$
    DECLARE
        _dbrid_auftg            varchar := null;
        _dbrid_lieferschein_pos varchar := null;
    BEGIN
        -- Ermittlung der Eigenschaften der Vorgänger und die des eigenen Datensatzes.
        -- mögliche Ausgangsobjekt:
          -- AR:  Ausgangsrechnung (belzeil)
          -- LFS: Lieferschein (lieferschein_pos)


        -- ohne Datensatz raus per STRICT

        -- Für AR die Vorgänger holen -> LFS -> Auftrag.
        IF _tname = 'belzeil' THEN
            -- LFS direkt an AR
            _dbrid_lieferschein_pos :=
                  belegpos.dbrid
                FROM belegpos
                    JOIN belzeil_grund ON bz_belp_id = belp_id
                WHERE belzeil_grund.dbrid = _dbrid
            ;

            -- Auftragsposition direkt an AR
            _dbrid_auftg :=
                  auftg.dbrid
                FROM auftg
                    JOIN belzeil_grund ON
                         bz_auftg    = ag_nr
                     AND bz_auftgpos = ag_pos
                WHERE ag_astat = 'E'
                  AND belzeil_grund.dbrid = _dbrid
            ;

        END IF;


        -- für LFS die Vorgänger holen -> Auftrag
        IF  _tname = 'lieferschein_pos' THEN
            -- Auftragsposition direkt am LFS
            _dbrid_auftg :=
                  auftg.dbrid
                FROM auftg
                    JOIN belegpos ON belp_ag_id = ag_id
                WHERE belegpos.dbrid = _dbrid
           ;

        END IF;

        -- Eigenschaften ermitteln und ausgeben.
        -- Alle RETURNs werden ausgeführt.

        -- Eigenschaften des Ausgangsobjekts
        RETURN QUERY
            SELECT * FROM TRecnoParam.get__all_as_rows__by__tablename_dbrid( _tname, _dbrid )
        ;

        -- Eigenschaften des Auftrags
        RETURN QUERY
            SELECT * FROM TRecnoParam.get__all_as_rows__by__tablename_dbrid( 'auftg', _dbrid_auftg )
            WHERE _dbrid_auftg IS NOT NULL
        ;

        -- Eigenschaften des LFS
        RETURN QUERY
            SELECT * FROM TRecnoParam.get__all_as_rows__by__tablename_dbrid( 'lieferschein_pos', _dbrid_lieferschein_pos )
            WHERE _dbrid_lieferschein_pos IS NOT NULL
        ;

    END $$ LANGUAGE plpgsql STRICT;
  --

  -- EAV Parameter pro Record aus Einzelzeilen zusammenaddiert als einzeiliger Wert für Anzeige in Record (f2) im Format "pnameX = value; pnameY = value"
  CREATE OR REPLACE FUNCTION TRecnoParam.get__all_as_text__by__tablename_dbrid(
       _tname varchar,
       _dbrid varchar
    ) RETURNS varchar AS $$
      -- kein coalesce um value => leere werden weggelassen
      SELECT
        string_agg(r_descr__reg_bez__as__fieldalias || ' = ' || r_value, '; ')
      FROM TRecnoParam.get__all_as_rows__by__tablename_dbrid(_tname, _dbrid)
      ;

    $$ LANGUAGE sql;
  --

  -- Ermöglicht zentralisierte Logik-Anpassung für automatisch angelegte Parametern #14614.
  CREATE OR REPLACE FUNCTION TRecnoParam.conditions_check(
        src_name    varchar,
        dest_name   varchar,
        src_value   varchar,
        dest_value  varchar
    ) RETURNS BOOL AS $$
    BEGIN

      RETURN
            src_name IS NULL
        OR  (     src_name  IS NOT DISTINCT FROM dest_name
              AND src_value IS NOT DISTINCT FROM dest_value
        )
      ;

    END $$ LANGUAGE plpgsql IMMUTABLE;
  --
--

-- WERTE SETZEN

  -- (Boolean-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.Set(IN pname varchar, IN rdbrid varchar, IN rValue boolean) RETURNS void AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, ifThen(rValue,'True', 'False')); -- Konvertierung: Analog cxGrid Editoren
    END $$ LANGUAGE plpgsql;
  --

  -- (Date-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.Set(IN pname varchar, IN rdbrid varchar, IN rValue DATE) RETURNS void AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, rValue :: text); -- Konvertierung: Analog cxGrid Editoren?
    END $$ LANGUAGE plpgsql;
  --

  -- (Integer-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.Set(IN pname varchar, IN rdbrid varchar, IN rValue integer) RETURNS void AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, rValue :: text ); -- Konvertierung: Analog cxGrid Editoren?
    END $$ LANGUAGE plpgsql;
  --

  -- (Numeric-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.Set(IN pname varchar, IN rdbrid varchar, IN rValue numeric) RETURNS void AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, rValue :: text ); -- Konvertierung: Analog cxGrid Editoren?
    END $$ LANGUAGE plpgsql;
  --

  -- (TimeStamp-Parameter)
  CREATE OR REPLACE FUNCTION TRecnoParam.Set(IN pname varchar, IN rdbrid varchar, IN rValue TIMESTAMP WITHOUT TIME ZONE) RETURNS void AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, rValue :: text ); -- Konvertierung: Analog cxGrid Editoren?
    END $$ LANGUAGE plpgsql;
  --

  -- (varchar /String-Parameter)
    CREATE OR REPLACE FUNCTION TRecnoParam.Set(
    IN pname      varchar,
    IN rdbrid     varchar,
    IN rValue     varchar(100),
    IN runit      varchar(100) DEFAULT NULL,
    IN rtablename varchar(50)  DEFAULT NULL,
    IN rdescr     varchar(100) DEFAULT NULL
    ) RETURNS void
  AS $$
    BEGIN
      PERFORM TRecnoParam.SetValue(pname, rdbrid, rValue::TEXT, TRUE, NULL, NULL, rtablename, runit, rdescr);
    END $$ LANGUAGE plpgsql;
  --

  -- [BASISFUNKTION, Update oder Insert] Legt den Parameter neu an und setzt den Wert auf den entsprechenden Defaultwert.
  -- Gibt den neu gesetzten Default als varchar zurück. _AllowUpdate=FALSE unterbindet Updates auf bestehende Datensätze.
  CREATE OR REPLACE FUNCTION TRecnoParam.SetDefault(
        _pname                  varchar,
        _rdbrid                 varchar,
        _AllowUpdate            boolean = true,
        _ErrorIfDefaultIsEmpty  boolean = false,
        _fieldname              varchar = null,
        _fieldvalue             varchar = null
    ) RETURNS text AS $$
    DECLARE
        defValue  text;
        pType     varchar;
    BEGIN
        IF coalesce( _rdbrid, '' ) = '' THEN
            RAISE EXCEPTION 'TRecnoParam.SetDefault: Missing IN-Parameter _rdbrid';
        END IF;

        SELECT
          reg_default,
          reg_paramtype
        INTO
          defValue,
          pType
        FROM recnogroup
        WHERE reg_pname = _pname
          AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
        ;


        IF      _ErrorIfDefaultIsEmpty
            AND coalesce( defValue, '' ) = ''
        THEN
            RAISE EXCEPTION 'TRecnoParam.SetDefault: Missing Default-Value for Parameter % (r_dbrid = %).', _pname, _rdbrid;
        END IF;


        IF coalesce( pType, '' ) = 'ptTEXT' THEN
            PERFORM
                TRecnoParam.SetText(
                    _pname,
                    _rdbrid,
                    defValue,
                    _AllowUpdate,
                    _fieldname,
                    _fieldvalue
                )
            ;

        ELSE
            PERFORM
                TRecnoParam.SetValue(
                    _pname,
                    _rdbrid,
                    defValue,
                    _AllowUpdate,
                    _fieldname,
                    _fieldvalue
                )
            ;

        END IF;


        RETURN defValue;
    END $$ LANGUAGE plpgsql;
  --

  -- [BASISFUNKTION, Update oder Insert] Wenn bei Update nichts getroffen wurde, wird der Satz neu angelegt.
  -- Setzt den Wert auf rValue. _AllowUpdate=FALSE unterbindet Updates auf bestehende Datensätze.
  CREATE OR REPLACE FUNCTION TRecnoParam.SetValue(
        _pname        varchar,
        _rdbrid       varchar,
        _rvalue       text,
        _AllowUpdate  boolean = true,
        _fieldname    varchar = null,
        _fieldvalue   varchar = null,
        _rtablename   varchar = null,
        _runit        varchar = null,
        _rdescr       varchar = null
    ) RETURNS void AS $$
    BEGIN
        IF _AllowUpdate THEN

            -- Update auf den entsprechenden Datensatz versuchen
            UPDATE recnokeyword SET
              r_value = _rvalue,
              r_unit  = _runit
            WHERE r_dbrid = _rdbrid
              AND r_reg_pname = _pname
              -- Auftragsbearbeitung LOLL: Parametername und Value sind Key (Artikel)
              AND CASE
                      WHEN _pname = 'auftg.bearbtg.AVORzust' THEN
                          r_value = _rvalue
                      ELSE true
                  END
            ;

            -- Wenn das nicht trifft, INSERT mit Wert rValue.
            IF NOT FOUND THEN
                IF _pname IS null THEN
                    INSERT INTO recnokeyword
                      ( r_reg_pname, r_tablename, r_dbrid, r_value, r_unit, r_kategorie, r_descr )
                    SELECT _pname  , _rtablename, _rdbrid, _rvalue, _runit, _rtablename, _rdescr;
                ELSE
                    INSERT INTO recnokeyword
                      ( r_reg_pname, r_tablename, r_dbrid, r_value, r_unit )
                    SELECT DISTINCT
                      _pname,
                      coalesce( _rtablename, reg_tablename ),
                      _rdbrid,
                      _rvalue,
                      coalesce( _runit, reg_m_iso )
                    FROM recnogroup
                    WHERE
                      -- doppelt angelegte Parameter (mit AC ohne AC) ausschließen
                      TSystem.Equals( reg_pname, _pname)
                      -- AC-Prüfung: Parameter entweder ohne AC, oder Parameter mit AC vergleichen mit AC (Funktionsangabe).
                      -- Wenn kein ac angg. ist, dann bei Artikel-Parametern automatisch vergleichen.
                      AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
                    ;
                 END IF;

            END IF;

        -- kein Update erlaubt
        -- Nur Insert und nur, wenn nicht vorhanden.
        ELSIF NOT EXISTS (SELECT true FROM recnokeyword WHERE r_dbrid = _rdbrid AND r_reg_pname = _pname) THEN

            INSERT INTO recnokeyword
              ( r_reg_pname, r_tablename, r_dbrid, r_value, r_unit ) -- INSERT vgl. oben
            SELECT DISTINCT
              _pname,
              coalesce( _rtablename, reg_tablename ),
              _rdbrid,
              _rvalue,
              coalesce( _runit, reg_m_iso )
            FROM recnogroup
            WHERE reg_pname = _pname
              AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
            ;

        END IF;


        RETURN;
    END $$ LANGUAGE plpgsql VOLATILE;
  --

  -- (Text-Parameter) Setzt das Feld r_txt aus Recnokeyword. Enthält entweder Bemerkung zu r_value oder ist selbst ein Parameter vom Typ Text.
  CREATE OR REPLACE FUNCTION TRecnoParam.SetText(
       -- TODO: Code komplett doppelt mit SetValue. Zusammenführen / besser machen
        _pname        varchar,
        _rdbrid       varchar,
        _rvalue       text,
        _AllowUpdate  boolean = true,
        _fieldname    varchar = null,
        _fieldvalue   varchar = null
    ) RETURNS void AS $$
    BEGIN
        IF _AllowUpdate THEN

            -- Update auf den entsprechenden Datensatz versuchen
            UPDATE recnokeyword SET
              r_txt = _rvalue
            WHERE r_dbrid = _rdbrid
              AND r_reg_pname = _pname
            ;

            -- wenn das nicht trifft, INSERT mit Wert rValue.
            IF NOT FOUND THEN

                INSERT INTO recnokeyword
                  ( r_reg_pname, r_tablename, r_dbrid, r_txt )
                SELECT DISTINCT
                  reg_pname,
                  reg_tablename,
                  _rdbrid,
                  _rvalue
                FROM recnogroup
                WHERE
                  -- doppelt angelegte Parameter (mit AC ohne AC) ausschließen
                      reg_pname = _pname
                  -- AC-Prüfung: Parameter entweder ohne AC, oder Parameter mit AC vergleichen mit ac (Funktionsangabe).
                  -- Wenn kein ac angg. ist, dann bei Artikel-Parametern automatisch vergleichen.
                  AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
                ;

            END IF;

        -- kein Update erlaubt
        -- Nur Insert und nur, wenn nicht vorhanden.
        ELSIF NOT EXISTS (SELECT true FROM recnokeyword WHERE r_dbrid = _rdbrid AND r_reg_pname = _pname) THEN

            INSERT INTO recnokeyword
              ( r_reg_pname, r_tablename, r_dbrid, r_txt ) -- INSERT vgl. oben
            SELECT DISTINCT
              reg_pname,
              reg_tablename,
              _rdbrid,
              _rvalue
            FROM recnogroup
            WHERE reg_pname = _pname
              AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
            ;

        END IF;


        RETURN;
    END $$ LANGUAGE plpgsql VOLATILE;
  --

-- HILFSFUNKTIONEN FÜR ENUMERATIONEN-PARAMETER (Value ist in Listenform über die Vorgaben definiert. Einträge aus der Liste werden bearbeitet im Value.

  -- (Enum-Parameter) - Gibt alle verfügbare Vorgabe-Enum-Werte eines Enum-Parameters zurück.
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_ValueList(
      _pname        varchar,
      _fieldname    varchar = null,
      _fieldvalue   varchar = null,

      OUT id        integer,
      OUT pos       integer,
      OUT code      varchar,
      OUT bez       varchar
    ) RETURNS SETOF record AS $$
    BEGIN
        RETURN QUERY
        SELECT
          rege_id,
          rege_pos,
          rege_code,
          rege_bez
        FROM RecnoEnums
          JOIN recnogroup ON reg_pname = rege_reg_pname
        WHERE
              reg_pname = _pname
          AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
        ORDER BY rege_pos, rege_id
        ;

    END $$ LANGUAGE plpgsql STABLE;
  --

  -- (Enum-Parameter) - Gibt Bezeichung eines Vorgabe-Enum-Wertes eines Enum-Parameters zurück.
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_ValueDescr(
        _pname        varchar,
        _code         varchar,
        _fieldname    varchar = null,
        _fieldvalue   varchar = null
    ) RETURNS varchar AS $$
    BEGIN
        RETURN
          rege_bez
        FROM RecnoEnums
          JOIN recnogroup ON reg_pname = rege_reg_pname
        WHERE
              reg_pname = _pname
          AND rege_code = _code
          AND TRecnoParam.conditions_check( reg_field_name, _fieldname, reg_field_value, _fieldvalue )
        ;

    END $$ LANGUAGE plpgsql STABLE;
  --
  
  --Wrapperfunktion um für Enum auch den Wert statt value des Enum-Idents zu zeigen
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_Descr(
      _pname varchar,
      _rdbrid varchar,
      _defaultValue varchar(100) DEFAULT NULL,
      _fieldname varchar DEFAULT NULL,
      _fieldvalue varchar DEFAULT NULL
      ) RETURNS varchar AS $$
      DECLARE
          enum_code varchar(100);
      BEGIN
          enum_code := TRecnoParam.GetEnum(_pname, _rdbrid, _defaultValue);

          RETURN TRecnoParam.Enum_ValueDescr(_pname, enum_code, _fieldname, _fieldvalue);
      END $$ LANGUAGE plpgsql STABLE;
  --
  
  --Erweiterung von Basisfunktion TRecnoParam.GetValue um für Enum auch den Wert statt value des Enum-Idents zu zeigen
  CREATE OR REPLACE FUNCTION TRecnoParam.GetValueDescr(
      _pname varchar,
      _rdbrid varchar,
      _defaultValue text DEFAULT NULL,
      _fieldname varchar DEFAULT NULL,
      _fieldvalue varchar DEFAULT NULL
      ) RETURNS text AS $$
      DECLARE
          val text;
      BEGIN
          -- Hole den gespeicherten Wert
          val := TRecnoParam.GetValue(_pname, _rdbrid, _defaultValue);

          -- Prüfe, ob es ein Enum ist und versuche, die Beschreibung zu holen
          RETURN COALESCE(
              TRecnoParam.Enum_ValueDescr(_pname, val, _fieldname, _fieldvalue),
          val
      );
      END $$ LANGUAGE plpgsql STABLE;
  --  

  -- (Enum-Parameter) - Prüft ob ein bestimmter Enum-Wert (also ein Flag) enthalten ist
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_ContainsFlag(IN pname varchar, IN rdbrid varchar, IN regeCode varchar) RETURNS boolean AS $$
    BEGIN
      RETURN TSystem.ENUM_GetValue(TRecnoParam.GetEnum(pname, rdbrid), regeCode);
    END $$ LANGUAGE plpgsql;
  --

  -- (Enum-Parameter [Flags]) - Setzt ein bestimmtes Flag, wenn es noch nicht enthalten ist.
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_AddFlag(IN pname varchar, IN rdbrid varchar, IN regeCode varchar) RETURNS VOID AS $$
    DECLARE
        enumValues  varchar;
        newValues   varchar;
    BEGIN
      enumValues := TRecnoParam.GetEnum(pname, rdbrid);
      newValues  := TSystem.ENUM_SetValue(enumValues, regeCode);

      IF enumValues IS DISTINCT FROM newValues THEN
          PERFORM TRecnoParam.Set(pname, rdbrid, newValues);
      END IF;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  -- (Enum-Parameter [Flags]) - Entfernt ein bestimmtes Flag, wenn es enthalten ist.
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_RemoveFlag(IN pname varchar, IN rdbrid varchar, IN regeCode varchar) RETURNS VOID AS $$
    DECLARE
        enumValues  varchar;
        newValues   varchar;
    BEGIN
      enumValues := TRecnoParam.GetEnum(pname, rdbrid);
      newValues  := TSystem.ENUM_DelValue(enumValues, regeCode);

      IF enumValues IS DISTINCT FROM newValues THEN
          PERFORM TRecnoParam.Set(pname, rdbrid, newValues);
      END IF;

      RETURN;
    END $$ LANGUAGE plpgsql;
  --

  -- (Enum-Parameter [Flags]) - Nimmt ein Enum-Flag in die Werte auf oder entfernt ihn, wenn er schon enthalten ist. Gibt True zurück,
  -- wenn der Wert aufgenommen wurde, false wenn er nicht mehr gesetzt ist.
  CREATE OR REPLACE FUNCTION TRecnoParam.Enum_FlipFlag(IN pname varchar, IN rdbrid varchar, IN regeCode varchar) RETURNS boolean AS $$
    DECLARE
        enumValues  varchar;
        result      boolean;
    BEGIN
      enumValues := TRecnoParam.GetEnum(pname, rdbrid);
      result     := TSystem.ENUM_GetValue(enumValues, regeCode);

      IF result THEN
          enumValues := TSystem.ENUM_DelValue(enumValues, regeCode);
      ELSE
          enumValues := TSystem.ENUM_SetValue(enumValues, regeCode);
      END IF;

      PERFORM TRecnoParam.Set(pname, rdbrid, enumValues);

      RETURN NOT result;
    END $$ LANGUAGE plpgsql;
  --
--

-- SONSTIGES
  -- #7135: Ein einzelner Parameter aus Quell-Datensatz wird mit gleichem Wert an Ziel-Datensatz angehangen.
  CREATE OR REPLACE FUNCTION TRecnoParam.Copy(IN pName varchar, IN srcDBRID varchar, IN trgDBRID varchar, IN trgTABLE varchar DEFAULT NULL) RETURNS boolean AS $$
    BEGIN
      INSERT INTO recnokeyword (r_tablename, r_dbrid, r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname)
      SELECT
        COALESCE(trgTABLE, r_tablename), trgDBRID, -- An Ziel-Datensatz kopieren. Ggf. abweichende Tabelle.
        r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname -- Daten übernehmen.
      FROM recnokeyword
      WHERE r_reg_pname = pName -- bestimmter Parameter
        AND r_dbrid = srcDBRID; -- aus Quell-Datensatz

      RETURN FOUND; -- Rückgabe, ob Kopieren erfolgt ist oder nicht.
    END $$ LANGUAGE plpgsql;
  --

  -- #7135: Alle Parameter einer bestimmten Gruppe aus Quell-Datensatz (z.Bsp. Durchlaufzeiten) werden mit gleichem Wert an Ziel-Datensatz angehangen.
  CREATE OR REPLACE FUNCTION TRecnoParam.CopyGroup(groupName varchar, srcDBRID varchar, trgDBRID varchar, IN trgTABLE varchar DEFAULT NULL) RETURNS integer AS $$
    DECLARE rows_processed integer;
    BEGIN
      INSERT INTO recnokeyword (r_tablename, r_dbrid, r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname)
      SELECT
        COALESCE(trgTABLE, r_tablename), trgDBRID, -- An Ziel-Datensatz kopieren. Ggf. abweichende Tabelle.
        r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname -- Daten übernehmen.
      FROM recnokeyword
        JOIN recnogroup ON reg_pname = r_reg_pname
      WHERE reg_gruppe = groupName  -- bestimmte Gruppe
        AND r_dbrid = srcDBRID     -- aus Quell-Datensatz
      ORDER by r_id;

      GET DIAGNOSTICS rows_processed = ROW_COUNT;

      RETURN rows_processed; -- Anzahl an Datensätze, welche kopiert wurden, ausgeben.
    END $$ LANGUAGE plpgsql;
  --

  -- #7135: Alle Parameter eines Quell-Datensatzes werden mit gleichem Wert an Ziel-Datensatz angehangen.
    -- Anmerkung: EAV-Parameter haben r_reg_pname gesetzt. Bei freien Parametern muss r_kategorie leer sein.
    -- Basiert darauf, dass PrintSettings, KeywordSearch etc. dort 'internal system usage' reinschreiben.
  CREATE OR REPLACE FUNCTION TRecnoParam.CopyAll(
        IN srcDBRID varchar,
        IN trgDBRID varchar,
        IN trgTABLE varchar DEFAULT NULL,
        IN _SystemParams boolean DEFAULT false
        )
        RETURNS integer
        AS $$
        DECLARE rows_processed integer;
        BEGIN
          INSERT INTO recnokeyword (r_tablename, r_dbrid, r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname)
          SELECT
            coalesce(trgTABLE, r_tablename), trgDBRID, -- An Ziel-Datensatz kopieren. Ggf. abweichende Tabelle.
            r_kategorie, r_descr, r_unit, r_txt, r_not, r_value, r_min, r_max, r_reg_pname -- Daten übernehmen.
          FROM recnokeyword
          WHERE r_dbrid = srcDBRID -- aus Quell-Datensatz
            AND ((r_reg_pname IS NOT NULL) OR (r_reg_pname IS NULL AND r_kategorie IS NULL)) -- definierte und freie Parameter
            AND (   _SystemParams
                 OR ( coalesce(r_reg_pname, '') NOT IN ('System.Printing', 'internal system usage') ) -- doppelte Printsetting siehe https://redmine.prodat-sql.de/issues/18768 -- Schlüsselworte kopieren?!
                )
          ORDER BY r_id;

          GET DIAGNOSTICS rows_processed = ROW_COUNT;

          RETURN rows_processed; -- Anzahl an Datensätze, welche kopiert wurden, ausgeben.
        END $$ LANGUAGE plpgsql;
  --

  -- #11475: Ein einzelner Parameter wird mit gleichem Wert aus Quell-Datensatz für Ziel-Datensatz aktualisiert.
    -- trgTABLE hier nicht zum Umschreiben angeboten, da Ziel schon korrekt sein muss.
  CREATE OR REPLACE FUNCTION TRecnoParam.Update(IN pName varchar, IN srcDBRID varchar, IN trgDBRID varchar) RETURNS boolean AS $$
    BEGIN
      UPDATE recnokeyword SET
        r_kategorie = src.r_kategorie,
        r_descr     = src.r_descr,
        r_unit      = src.r_unit,
        r_txt       = src.r_txt,
        r_not       = src.r_not,
        r_value     = src.r_value,
        r_min       = src.r_min,
        r_max       = src.r_max
      FROM recnokeyword AS src
      WHERE src.r_reg_pname = pName -- bestimmter Parameter
        AND src.r_dbrid = srcDBRID  -- aus Quell-Datensatz
        AND recnokeyword.r_reg_pname = pName  -- an identischen Parameter
        AND recnokeyword.r_dbrid = trgDBRID   -- am Ziel-Datensatz
        -- es gibt Unterschiede in Daten
        AND (  recnokeyword.r_kategorie IS DISTINCT FROM src.r_kategorie
            OR recnokeyword.r_descr     IS DISTINCT FROM src.r_descr
            OR recnokeyword.r_unit      IS DISTINCT FROM src.r_unit
            OR recnokeyword.r_txt       IS DISTINCT FROM src.r_txt
            OR recnokeyword.r_not       IS DISTINCT FROM src.r_not
            OR recnokeyword.r_value     IS DISTINCT FROM src.r_value
            OR recnokeyword.r_min       IS DISTINCT FROM src.r_min
            OR recnokeyword.r_max       IS DISTINCT FROM src.r_max
        )
      ;

      RETURN FOUND; -- Rückgabe, ob Update erfolgt ist oder nicht.
    END $$ LANGUAGE plpgsql;
  --

  -- #11475: Alle Parameter einer bestimmten Gruppe werden mit gleichem Wert aus Quell-Datensatz am Ziel-Datensatz aktualisiert.
    -- trgTABLE hier nicht zum Umschreiben angeboten, da Ziel schon korrekt sein muss.
  CREATE OR REPLACE FUNCTION TRecnoParam.UpdateGroup(groupName varchar, srcDBRID varchar, trgDBRID varchar) RETURNS integer AS $$
    DECLARE rows_processed integer;
    BEGIN
      UPDATE recnokeyword SET
        r_kategorie = src.r_kategorie,
        r_descr     = src.r_descr,
        r_unit      = src.r_unit,
        r_txt       = src.r_txt,
        r_not       = src.r_not,
        r_value     = src.r_value,
        r_min       = src.r_min,
        r_max       = src.r_max
      FROM recnokeyword AS src
        JOIN recnogroup ON reg_pname = src.r_reg_pname
      WHERE reg_gruppe = groupName  -- bestimmte Gruppe
        AND src.r_dbrid = srcDBRID  -- aus Quell-Datensatz
        AND recnokeyword.r_reg_pname = src.r_reg_pname  -- an identischen Parameter(n)
        AND recnokeyword.r_dbrid = trgDBRID             -- am Ziel-Datensatz
        -- es gibt Unterschiede in Daten
        AND (  recnokeyword.r_kategorie IS DISTINCT FROM src.r_kategorie
            OR recnokeyword.r_descr     IS DISTINCT FROM src.r_descr
            OR recnokeyword.r_unit      IS DISTINCT FROM src.r_unit
            OR recnokeyword.r_txt       IS DISTINCT FROM src.r_txt
            OR recnokeyword.r_not       IS DISTINCT FROM src.r_not
            OR recnokeyword.r_value     IS DISTINCT FROM src.r_value
            OR recnokeyword.r_min       IS DISTINCT FROM src.r_min
            OR recnokeyword.r_max       IS DISTINCT FROM src.r_max
        )
      ;

      GET DIAGNOSTICS rows_processed = ROW_COUNT;

      RETURN rows_processed; -- Anzahl an Datensätze, welche kopiert wurden, ausgeben.
    END $$ LANGUAGE plpgsql;
  --

  -- #11475: Alle Parameter eines Quell-Datensatz werden mit gleichem Wert am Ziel-Datensatz aktualisiert.
    -- trgTABLE hier nicht zum Umschreiben angeboten, da Ziel schon korrekt sein muss.
    -- Anmerkung: EAV-Parameter haben r_reg_pname gesetzt, bei freien Parametern muss r_kategorie leer sein.
    -- Basiert darauf, dass PrintSettings, KeywordSearch etc. dort 'internal system usage' reinschreiben.
  CREATE OR REPLACE FUNCTION TRecnoParam.UpdateAll(srcDBRID varchar, trgDBRID varchar) RETURNS integer AS $$
    DECLARE rows_processed integer;
    BEGIN
      UPDATE recnokeyword SET
        r_kategorie = src.r_kategorie,
        r_descr     = src.r_descr,
        r_unit      = src.r_unit,
        r_txt       = src.r_txt,
        r_not       = src.r_not,
        r_value     = src.r_value,
        r_min       = src.r_min,
        r_max       = src.r_max
      FROM recnokeyword AS src
      WHERE src.r_dbrid = srcDBRID  -- aus Quell-Datensatz
        AND ((src.r_reg_pname IS NOT NULL) OR (src.r_reg_pname IS NULL AND src.r_kategorie IS NULL)) -- definierte und freie Parameter
        AND recnokeyword.r_dbrid = trgDBRID               -- am Ziel-Datensatz
        AND ((recnokeyword.r_reg_pname = src.r_reg_pname) -- an identischen Parameter(n)
             -- freie
             OR (             src.r_reg_pname IS NULL AND          src.r_kategorie IS NULL
                 AND recnokeyword.r_reg_pname IS NULL AND recnokeyword.r_kategorie IS NULL
                 AND recnokeyword.r_descr = src.r_descr -- Ident über Beschreibung
             )
        )
        -- es gibt Unterschiede in Daten
        AND (  recnokeyword.r_kategorie IS DISTINCT FROM src.r_kategorie
            OR recnokeyword.r_descr     IS DISTINCT FROM src.r_descr
            OR recnokeyword.r_unit      IS DISTINCT FROM src.r_unit
            OR recnokeyword.r_txt       IS DISTINCT FROM src.r_txt
            OR recnokeyword.r_not       IS DISTINCT FROM src.r_not
            OR recnokeyword.r_value     IS DISTINCT FROM src.r_value
            OR recnokeyword.r_min       IS DISTINCT FROM src.r_min
            OR recnokeyword.r_max       IS DISTINCT FROM src.r_max
        )
      ;

      GET DIAGNOSTICS rows_processed = ROW_COUNT;

      RETURN rows_processed; -- Anzahl an Datensätze, welche kopiert wurden, ausgeben.
    END $$ LANGUAGE plpgsql;
  --

  -- #11475: Ein einzelner Parameter an Ziel-Datensatz wird gelöscht.
  CREATE OR REPLACE FUNCTION TRecnoParam.Delete(IN pName varchar, IN trgDBRID varchar) RETURNS boolean AS $$
    BEGIN
      DELETE FROM recnokeyword
      WHERE r_reg_pname = pName -- bestimmter Parameter
        AND r_dbrid = trgDBRID; -- an Ziel-Datensatz

      RETURN FOUND; -- Rückgabe, ob Löschen erfolgt ist oder nicht.
    END $$ LANGUAGE plpgsql;
  --

  -- #11475: Alle Parameter einer bestimmten Gruppe an Ziel-Datensatz werden gelöscht.
  CREATE OR REPLACE FUNCTION TRecnoParam.DeleteGroup(groupName varchar, trgDBRID varchar) RETURNS integer AS $$
    DECLARE rows_processed integer;
    BEGIN
      DELETE FROM recnokeyword
      WHERE EXISTS(SELECT true FROM recnogroup WHERE reg_pname = r_reg_pname AND reg_gruppe = groupName)  -- bestimmte Gruppe
        AND r_dbrid = trgDBRID;                                                                           -- an Ziel-Datensatz

      GET DIAGNOSTICS rows_processed = ROW_COUNT;

      RETURN rows_processed; -- Anzahl an Datensätze, welche gelöscht wurden, ausgeben.
    END $$ LANGUAGE plpgsql;
  --

  -- #11475: Alle Parameter des Ziel-Datensatzes werden gelöscht.
    -- Anmerkung: EAV-Parameter haben r_reg_pname gesetzt, bei freien Parametern muss r_kategorie leer sein.
    -- Basiert darauf, dass PrintSettings, KeywordSearch etc. dort alle 'internal system usage' reinschreiben.
  CREATE OR REPLACE FUNCTION TRecnoParam.DeleteAll(trgDBRID varchar) RETURNS integer AS $$
    DECLARE rows_processed integer;
    BEGIN
      DELETE FROM recnokeyword
      WHERE r_dbrid = trgDBRID -- an Ziel-Datensatz
        AND ((r_reg_pname IS NOT NULL) OR (r_reg_pname IS NULL AND r_kategorie IS NULL)); -- definierte und freie Parameter

      GET DIAGNOSTICS rows_processed = ROW_COUNT;

      RETURN rows_processed; -- Anzahl an Datensätze, welche kopiert wurden, ausgeben.
    END $$ LANGUAGE plpgsql;
  --

  -- Parameter aus Stammdaten in Auftrag kopieren, siehe #11513.
  -- Freie Gruppe derzeit nicht unterstützt (nur per _gruppen = '%').
  -- Setting: Liefert Parameter (reg_pname) für Zusatzadresse, aus der weitere Eigenschaften kopiert werden sollen.
    -- TSystem.Settings__Set('TRecnoParam.auftg__params_sync.Zusatzadresse', 'auftg.param.Endkunde'),
  -- DROP FUNCTION IF EXISTS TRecnoParam.auftg__params_sync(integer, varchar, varchar, boolean, boolean, boolean, boolean, boolean);
  CREATE OR REPLACE FUNCTION TRecnoParam.auftg__params_sync(IN _ag_id integer, IN _gruppen varchar, IN _tabname varchar,  IN _aktualisieren boolean, IN _hinzufuehgen boolean, IN _loeschen boolean,  IN _alleHPos boolean, IN _InklUnterPos boolean) RETURNS void AS $$
    DECLARE
        rec                 record;
        agastat             varchar; -- Auftragsstatus von markierter Auftragsposition
        agnr                varchar; -- Auftragsnummer von markierter Auftragsposition
        agpos               varchar; -- Position von markierter Auftragsposition
        aghpos              varchar; -- Bezugsposition von markierter Auftragsposition
        group_to_handle     varchar; -- spez. Gruppen aus Stringliste _gruppen
        pname_add_address   varchar; -- Zusatzadresse aus Setting, s.o.
    BEGIN
      -- Fürs Debugging
      -- RETURN NEXT concat_ws('', '1  - ', 'Eingagsparameter', '_ag_id = ', _ag_id, ',  _art_gruppe = ', _art_gruppe, ',  _adk_gruppe = ', _adk_gruppe, ',  _InklUnterPos = ', _InklUnterPos, ',  _AuftgKomplett = ', _AuftgKomplett);

      SELECT ag_astat, ag_nr, ag_pos, ag_hpos
      INTO agastat, agnr, agpos, aghpos
      FROM auftg
      WHERE ag_id = _ag_id;

      --- Artikeleigentschaften kopieren
      IF _tabname = 'art' AND _gruppen IS NOT NULL THEN
          FOR rec IN (
              SELECT art.dbrid AS art_DBRID_Quelle, auftg.dbrid AS agDBRID_Ziel, ag_id, ag_nr, ag_pos, ag_aknr, ag_hpos
              FROM auftg
                JOIN art ON ak_nr = ag_aknr
              WHERE ag_astat = agastat AND ag_nr = agnr
                AND NOT ag_storno
                AND CASE WHEN     _alleHPos AND     _inklUnterPos THEN true -- Alle Hauptpositionen inkl. deren Unterpositionen
                         WHEN     _alleHPos AND NOT _inklUnterPos THEN ag_hpos IS NULL -- Nur alle Hauptpositionen (ohne Bezug zu anderer Position)
                         WHEN NOT _alleHPos AND     _inklUnterPos THEN ag_hpos = agpos OR ag_pos = agpos -- Selektierte Position und deren Unterpositionen
                         WHEN NOT _alleHPos AND NOT _inklUnterPos THEN ag_pos = agpos -- Nur selektierte Position
                     END
              ORDER BY ag_pos
          ) LOOP
              IF _gruppen = '%' THEN
                  IF _aktualisieren THEN
                      PERFORM TRecnoParam.UpdateAll(rec.art_DBRID_Quelle, rec.agDBRID_Ziel);
                  END IF;
                  IF _hinzufuehgen THEN
                      PERFORM TRecnoParam.CopyAll(rec.art_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');
                  END IF;
              ELSE
                  FOR group_to_handle IN SELECT trim(regexp_split_to_table(_gruppen, ','), '"') LOOP
                      IF _aktualisieren THEN
                          PERFORM TRecnoParam.UpdateGroup(group_to_handle, rec.art_DBRID_Quelle, rec.agDBRID_Ziel);
                      END IF;
                      IF _hinzufuehgen THEN
                          PERFORM TRecnoParam.CopyGroup(group_to_handle, rec.art_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');
                      END IF;
                      IF _loeschen AND NOT _aktualisieren AND NOT _hinzufuehgen THEN
                          PERFORM TRecnoParam.DeleteGroup(group_to_handle, rec.agDBRID_Ziel);
                      END IF;
                  END LOOP;
              END IF;

              -- Fürs Debugging
              -- RETURN NEXT concat_ws('', '2  - ', 'agDBRID_Ziel = ', rec.agDBRID_Ziel, ',  rec.ag_nr = ', rec.ag_nr, ',  rec.ag_pos = ', rec.ag_pos, ', ag_hpos = ', rec.ag_hpos, ',  rec.art_DBRID_Quelle = ', rec.art_DBRID_Quelle::varchar, ',  rec.ag_aknr = ', rec.ag_aknr);
          END LOOP;
      END IF;

      --- Adresseneigentschaften kopieren
      IF _tabname = 'adk' AND _gruppen IS NOT NULL THEN
          IF aghpos IS NOT NULL AND NOT _alleHPos THEN
              RAISE EXCEPTION '%', lang_text(29417); -- 'Kopieren ist unmöglich. Ausgewähle Position ist keine Hauptposition.'
          END IF;

          pname_add_address:= nullif(trim(TSystem.Settings__Get('TRecnoParam.auftg__params_sync.Zusatzadresse')), '');

          FOR rec IN
              SELECT adk.dbrid AS adk_DBRID_Quelle, auftg.dbrid AS agDBRID_Ziel, adk_zusatzadresse_DBRID_Quelle, ag_id, ag_nr, ag_pos, ag_aknr, ag_hpos
              FROM auftg
                JOIN adk ON ad_krz = ag_lkn
                LEFT JOIN LATERAL (
                    SELECT adk_zusatzadresse.dbrid AS adk_zusatzadresse_DBRID_Quelle
                    FROM adk AS adk_zusatzadresse
                    WHERE adk_zusatzadresse.ad_krz = (
                        SELECT r_value
                        FROM recnokeyword
                        WHERE r_tablename = 'auftg' AND r_dbrid = auftg.dbrid
                          AND r_reg_pname = pname_add_address
                        ORDER BY r_id DESC LIMIT 1)
                ) AS parameter_zusatz ON pname_add_address IS NOT NULL
              WHERE ag_astat = agastat AND ag_nr = agnr
                AND NOT ag_storno
                AND CASE WHEN     _alleHPos THEN ag_hpos IS NULL -- Nur alle Hauptpositionen (ohne Bezug zu anderer Position)
                         WHEN NOT _alleHPos THEN ag_pos = agpos AND ag_hpos IS NULL -- Nur selektierte Hauptposition
                    END
              ORDER BY ag_pos
          LOOP
              IF _gruppen = '%' THEN
                  IF _aktualisieren THEN
                      PERFORM TRecnoParam.UpdateAll(rec.adk_DBRID_Quelle, rec.agDBRID_Ziel);

                      IF rec.adk_zusatzadresse_DBRID_Quelle IS NOT NULL THEN
                          PERFORM TRecnoParam.UpdateAll(rec.adk_zusatzadresse_DBRID_Quelle, rec.agDBRID_Ziel);
                      END IF;
                  END IF;
                  IF _hinzufuehgen THEN
                      PERFORM TRecnoParam.CopyAll(rec.adk_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');

                      IF rec.adk_zusatzadresse_DBRID_Quelle IS NOT NULL THEN
                          PERFORM TRecnoParam.CopyAll(rec.adk_zusatzadresse_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');
                      END IF;
                  END IF;
              ELSE
                  FOR group_to_handle IN SELECT trim(regexp_split_to_table(_gruppen, ','), '"') LOOP
                      IF _aktualisieren THEN
                          PERFORM TRecnoParam.UpdateGroup(group_to_handle, rec.adk_DBRID_Quelle, rec.agDBRID_Ziel);

                          IF rec.adk_zusatzadresse_DBRID_Quelle IS NOT NULL THEN
                              PERFORM TRecnoParam.UpdateGroup(group_to_handle, rec.adk_zusatzadresse_DBRID_Quelle, rec.agDBRID_Ziel);
                          END IF;
                      END IF;
                      IF _hinzufuehgen THEN
                          PERFORM TRecnoParam.CopyGroup(group_to_handle, rec.adk_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');

                          IF rec.adk_zusatzadresse_DBRID_Quelle IS NOT NULL THEN
                              PERFORM TRecnoParam.CopyGroup(group_to_handle, rec.adk_zusatzadresse_DBRID_Quelle, rec.agDBRID_Ziel, 'auftg');
                          END IF;
                      END IF;
                      IF _loeschen AND NOT _aktualisieren AND NOT _hinzufuehgen THEN
                          PERFORM TRecnoParam.DeleteGroup(group_to_handle, rec.agDBRID_Ziel);
                      END IF;
                  END LOOP;
              END IF;

              -- Fürs Debugging
              -- RETURN NEXT concat_ws('', '2  - ','agDBRID_Ziel = ', rec.agDBRID_Ziel, ',  rec.ag_nr = ', rec.ag_nr, ',  rec.ag_pos = ', rec.ag_pos, ',   ag_hpos = ', rec.ag_hpos, ',  rec.adk_DBRID_Quelle = ', rec.adk_DBRID_Quelle::VARCHAR, ',  rec.ag_aknr = ',rec.ag_aknr);
          END LOOP;
      END IF;

      RETURN;
    END $$ LANGUAGE plpgsql VOLATILE STRICT;
  --

  -- Parameter aus Auftrag in ABK-AG kopieren, siehe #11514.
  -- Freie Gruppe derzeit nicht unterstützt (nur per _gruppen = '%').
  -- DROP FUNCTION IF EXISTS TRecnoParam.ab2__params_sync__from_auftg(integer, integer, VARCHAR, VARCHAR, boolean, boolean, boolean);
  CREATE OR REPLACE FUNCTION TRecnoParam.ab2__params_sync__from_auftg(IN _a2_id integer, IN _ag_id integer, IN _gruppen varchar, IN _tabname varchar,  IN _aktualisieren boolean, IN _hinzufuehgen boolean, IN _loeschen boolean) RETURNS void AS $$
    DECLARE
        rec             record;
        agastat         varchar;
        agnr            varchar;
        agpos           integer;
        ab2_DBRID_Ziel  varchar;
        group_to_handle varchar; -- spez. Gruppen aus Stringliste _gruppen
    BEGIN
      IF _ag_id IS NULL OR _gruppen IS NULL OR _tabname IS NULL OR _aktualisieren IS NULL OR _hinzufuehgen IS NULL OR _loeschen IS NULL THEN RETURN; END IF;

      SELECT ag_astat, ag_nr, ag_pos
      INTO agastat, agnr, agpos
      FROM auftg WHERE ag_id = _ag_id; -- Quell-Datensatz

      IF _tabname = 'ab2' AND _gruppen IS NOT NULL THEN
          SELECT dbrid INTO ab2_DBRID_Ziel FROM ab2 WHERE a2_id = _a2_id; -- Ziel-Datensatz

          FOR rec IN
              SELECT auftg.dbrid AS auftg_DBRID_Quelle
              FROM auftg
              WHERE (ag_id = _ag_id OR (ag_hpos = agpos AND ag_nr = agnr AND ag_astat = agastat)) -- Quell-Auftragsposition und zugehörige Unterpositionen
                AND NOT ag_storno
              ORDER BY ag_pos
          LOOP
              IF _gruppen = '%' THEN
                  IF _aktualisieren THEN
                      PERFORM TRecnoParam.UpdateAll(rec.auftg_DBRID_Quelle, ab2_DBRID_Ziel);
                  END IF;
                  IF _hinzufuehgen THEN
                      PERFORM TRecnoParam.CopyAll(rec.auftg_DBRID_Quelle, ab2_DBRID_Ziel, 'ab2');
                  END IF;
              ELSE
                  FOR group_to_handle IN SELECT trim(regexp_split_to_table(_gruppen, ','), '"') LOOP
                      IF _aktualisieren THEN
                          PERFORM TRecnoParam.UpdateGroup(group_to_handle, rec.auftg_DBRID_Quelle, ab2_DBRID_Ziel);
                      END IF;
                      IF _hinzufuehgen THEN
                          PERFORM TRecnoParam.CopyGroup(group_to_handle, rec.auftg_DBRID_Quelle, ab2_DBRID_Ziel, 'ab2');
                      END IF;
                      IF _loeschen AND NOT _aktualisieren AND NOT _hinzufuehgen THEN
                          PERFORM TRecnoParam.DeleteGroup(group_to_handle, ab2_DBRID_Ziel);
                      END IF;
                  END LOOP;
              END IF;
          END LOOP;
      END IF;

      RETURN;
    END $$ LANGUAGE plpgsql VOLATILE;
  --
--- #22251
CREATE OR REPLACE FUNCTION trecnoparam.RecNoGroup__insert_script__create(IN _reg_id integer) RETURNS text AS $$
DECLARE
    rec          record;
    rec_enums    record;
    _result      text = '';
    _columns_rg  text;
    _json_result jsonb;
BEGIN
    -- Dynamisch die Spalten ohne 'reg_id' auswählen; reg_id ist im Kundensystem immer anders
    SELECT string_agg(column_name, ', ')
    INTO _columns_rg
    FROM information_schema.columns
    WHERE table_name = 'recnogroup' AND column_name != 'reg_id';

    FOR rec IN
        SELECT reg_id, reg_pname
        FROM RecNoGroup
        WHERE reg_id = _reg_id
    LOOP
        -- Wir übergeben alle Spalten ausser 'reg_id'
        EXECUTE format('SELECT row_to_json(rn) FROM (SELECT %s FROM RecNoGroup WHERE reg_id = $1) AS rn', _columns_rg)
        INTO _json_result
        USING rec.reg_id;

        -- JSON-Ergebnis in _result einfügen
        _result = _result || tsystem.insert_into__from__json__in_line(
                                     'RecNoGroup',
                                    _json_result::json)
                || chr(10) || '---' || chr(10) || chr(10);

        IF EXISTS(SELECT true FROM RecNoEnums WHERE rege_reg_pname = rec.reg_pname) THEN
            FOR rec_enums IN
                SELECT rege_id, rege_reg_pname
                FROM RecNoEnums
                WHERE rege_reg_pname = rec.reg_pname
            LOOP
                _result = _result || (
                    SELECT tsystem.insert_into__from__json__in_line(
                        'RecNoEnums',
                        (
                            SELECT row_to_json(re) AS result
                            FROM (
                                SELECT rege_reg_pname, rege_pos, rege_code, rege_bez FROM RecNoEnums WHERE rege_id = rec_enums.rege_id
                            ) AS re
                        )
                    )
                ) || chr(10) || '---' || chr(10);
            END LOOP; -- rec_enums
        END IF;

    END LOOP; -- rec

    RETURN _result;
END $$ LANGUAGE plpgsql STABLE;
--


-- Funktion zum (Wieder-)Erstellen eines Crosstab ähnlichen VIEWs pro Tablename und Gruppe
  -- über IN color kann die Farbigkeit aller Spalten der Gruppe festgelegt werden
  -- der Paramtername wird implizit als Fieldalias angesehen
CREATE OR REPLACE FUNCTION TRecnoParam.pivot_view__by__tablename_reggroup__recreate(
    tname varchar,
    rgroup varchar,
    color varchar -- Farbige Dartsellung der Spalten
    ) RETURNS VOID AS $$
    DECLARE
      _func_name varchar;
      _view_name varchar;
      _column_names text;
      _column_paramcall text;
      _signature text;
      _cimformats text;
      _type varchar;
      _source_sql text;
      _query text;
      _query_view text;

    BEGIN
        -- Den Namen des zu erzeugenden VIEW bestimmen und Löschen
        _view_name := 'pivot__by__' || quote_ident(tname) ||'_'|| quote_ident(lower(rgroup)) || '__view';
        EXECUTE 'DROP VIEW IF EXISTS TRecnoParam.' || _view_name || '';

        -- Den Namen der zu erzeugenden Funktion bestimmen und Löschen
        _func_name := 'TRecnoParam.pivot__by__' || quote_ident(tname) ||'_'|| quote_ident(lower(rgroup)) || '__view__func';
        EXECUTE 'DROP FUNCTION IF EXISTS ' || _func_name || '(varchar)';

        -- Abschnitt dynamische Parameter Funktion erstellen

        -- Die Parameternamen als Liste der Spaltennamen für das SQL-Create-Function-Kommando vorbereiten
        WITH _columns AS (
            SELECT DISTINCT
                    reg_bez AS descr,
                    reg_pname AS col, -- der Parametername wird als FA verwendet
                    reg_pos,
                    reg_bez,
                    pt_dbtyp,
                    CASE WHEN (pt_dbtyp = 'boolean') THEN
                        'False'
                      ELSE
                        ''
                    END AS default_str -- Für default-Ausgabe Bool-Felder null als False
            FROM recnogroup
              JOIN TRecnoParam.SupportedTypes() ON pt_typ = reg_paramtype -- für Zuweisung des korrekten Datentyps der Spalte
            WHERE
              reg_tablename = tname
              AND  reg_gruppe = rgroup
              AND  coalesce(reg_visible, TRUE)
            ORDER BY reg_pos,reg_bez
        )
        SELECT --
          coalesce( string_agg( E'\n                OUT ' || quote_ident(col)   || '  ' || pt_dbtyp || '', ',' ), '' )    AS fsignature,  -- Leerzeichen für Formatierung der Ausgabe!
          coalesce( string_agg( E'\n                            ' || quote_ident(col) || '', ',' ), '' )    AS column_names,
          coalesce( string_agg( E'\n                            TRecnoParam.GetValueDescr(' || quote_literal(col) || ', _tnamedbrid, '|| quote_literal(default_str) ||') ', ',' ), '' )    AS column_paramcall,
          coalesce( 'concat_ws('', '',' ||string_agg( E'\n                            ' || quote_literal( col || '_' || color ), ',' ) || ')', '' ) AS cimformats
        INTO _signature, _column_names, _column_paramcall, _cimformats
        FROM _columns;

        -- Das innere SQL-Statement für Ausgabe-Funktion vorbereiten
        -- es wird statt der Parameter (recnokeyword) nur regnogroup geholt und per Funktion der Parameter-Value ~TRecnoParam.GetValue
           -- Grund ist die Ausgabe False für Boolean-Spalten, wenn nicht in recnokeyword zu tname.dbrid vorhanden, da Filter in Oberflächen auf false gesetzt werden
        _source_sql := $s1$
                        SELECT
                            $s1$ || _cimformats || $s2$ AS cimformat, $s2$ || _column_paramcall || $s3$
                        INTO
                            cimformat, $s3$ || _column_names || $s4$
                        ;
                       $s4$;

        -- Das SQL-Kommando zum Erzeugen der Function zusammenbauen.
        _query := $q1$
            CREATE OR REPLACE FUNCTION $q1$ || _func_name || $q2$ (
                IN _tnamedbrid varchar,
                OUT cimformat text, $q2$ || _signature || $q3$
                )
                RETURNS record AS $BODY$
                BEGIN $q3$ || _source_sql || $q4$
                  RETURN;
                END $BODY$ LANGUAGE plpgsql;
                $q4$;

        EXECUTE _query;

        -- Abschnitt VIEW auf dynamische Parameter Funktion erstellen
        -- um für die Oberflächen den Tablename zur Verfügung zu haben (mit Funktion nicht möglich)
        -- zusätzlich den Table tname aufnehmen: Grund ist die Ausgabe False für Boolean-Spalten; sonst wäre der Aufruf mit LEFT JOIN des VIEW null >> Filter nicht sinnvoll verwendbar
        _query_view := $q1$
            CREATE OR REPLACE VIEW TRecnoParam.$q1$ || _view_name || $q2$ AS
              (SELECT $q2$ || _view_name || $q3$__func.*, $q3$ || tname || $q4$.dbrid AS tbl_dbrid
               FROM $q4$ || tname || $q5$
               CROSS JOIN TRecnoParam.$q5$ || _view_name || $q6$__func($q6$ || tname || $q7$.dbrid) )
              ;
                $q7$; -- Kommentar nur wegen Sublime $q3$

        EXECUTE _query_view;

        IF (current_user = 'docker') OR (current_user = 'postgres') THEN
          RAISE NOTICE E'%: %', _func_name, _query;
          RAISE NOTICE E'%', _query_view;
        END IF;

    END $$
    LANGUAGE plpgsql SECURITY DEFINER;
    --
--
